<?php

namespace App\Listeners;

use Illuminate\Auth\Events\Failed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use App\Services\NotificacaoService;
use App\Mail\TentativasLoginNotificacao;

class LogTentativaLogin
{
    /**
     * Número máximo de tentativas falhas antes de disparar o alerta
     */
    const MAX_TENTATIVAS = 5;
    
    /**
     * Janela de tempo em minutos para considerar tentativas consecutivas
     */
    const JANELA_TEMPO = 30;
    
    /**
     * Create the event listener.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Handle the event.
     *
     * @param  Failed  $event
     * @return void
     */
    public function handle(Failed $event)
    {
        $email = $event->credentials['email'] ?? 'unknown';
        $ip = request()->ip();
        
        // Registra a tentativa no banco de dados
        $this->registrarTentativa($email, $ip);
        
        // Verifica se há muitas tentativas do mesmo IP
        $this->verificarTentativasIP($ip);
        
        // Verifica se há muitas tentativas para o mesmo email
        $this->verificarTentativasEmail($email);
    }
    
    /**
     * Registra a tentativa de login no banco
     *
     * @param string $email
     * @param string $ip
     * @return void
     */
    private function registrarTentativa($email, $ip)
    {
        DB::table('login_falhas')->insert([
            'email' => $email,
            'ip' => $ip,
            'user_agent' => request()->userAgent(),
            'created_at' => now()
        ]);
        
        Log::warning('Falha de login detectada', [
            'email' => $email,
            'ip' => $ip,
            'user_agent' => request()->userAgent()
        ]);
    }
    
    /**
     * Verifica se há muitas tentativas do mesmo IP
     *
     * @param string $ip
     * @return void
     */
    private function verificarTentativasIP($ip)
    {
        // Contagem de tentativas do mesmo IP nos últimos minutos
        $count = DB::table('login_falhas')
            ->where('ip', $ip)
            ->where('created_at', '>=', now()->subMinutes(self::JANELA_TEMPO))
            ->count();
            
        if ($count >= self::MAX_TENTATIVAS) {
            // Verifica se já enviamos um alerta recentemente (evita spam)
            $cacheKey = 'login_alert_ip_' . md5($ip);
            if (!Cache::has($cacheKey)) {
                $this->enviarAlerta('IP', $ip, $count);
                
                // Cache para não enviar alertas repetidos no período
                Cache::put($cacheKey, true, now()->addMinutes(60));
            }
        }
    }
    
    /**
     * Verifica se há muitas tentativas para o mesmo email
     *
     * @param string $email
     * @return void
     */
    private function verificarTentativasEmail($email)
    {
        if ($email == 'unknown') {
            return;
        }
        
        // Contagem de tentativas do mesmo email nos últimos minutos
        $count = DB::table('login_falhas')
            ->where('email', $email)
            ->where('created_at', '>=', now()->subMinutes(self::JANELA_TEMPO))
            ->count();
            
        if ($count >= self::MAX_TENTATIVAS) {
            // Verifica se já enviamos um alerta recentemente (evita spam)
            $cacheKey = 'login_alert_email_' . md5($email);
            if (!Cache::has($cacheKey)) {
                $this->enviarAlerta('Email', $email, $count);
                
                // Cache para não enviar alertas repetidos no período
                Cache::put($cacheKey, true, now()->addMinutes(60));
            }
        }
    }
    
    /**
     * Envia um alerta de tentativas múltiplas de login
     *
     * @param string $tipo
     * @param string $valor
     * @param int $count
     * @return void
     */
    private function enviarAlerta($tipo, $valor, $count)
    {
        // Busca as últimas tentativas
        $tentativas = DB::table('login_falhas')
            ->where($tipo == 'IP' ? 'ip' : 'email', $valor)
            ->where('created_at', '>=', now()->subMinutes(self::JANELA_TEMPO))
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
            
        $dados = [
            'tipo' => $tipo,
            'valor' => $valor,
            'count' => $count,
            'janela_tempo' => self::JANELA_TEMPO,
            'tentativas' => $tentativas,
            'data' => now()->format('d/m/Y H:i:s')
        ];
        
        // Log de segurança
        Log::channel('security')->alert('Possível tentativa de invasão detectada', $dados);
        
        try {
            // Obtém a lista de destinatários
            $destinatarios = NotificacaoService::getDestinatarios();
            
            // Envia notificação para cada destinatário
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new TentativasLoginNotificacao($dados));
            }
        } catch (\Exception $e) {
            Log::error('Falha ao enviar notificação de tentativas de login', [
                'erro' => $e->getMessage()
            ]);
        }
    }
} 