<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Inventario;
use App\Models\ItemInventario;
use App\Models\Produto;
use App\Models\User;
use App\Utils\EstoqueUtil;
use Illuminate\Support\Facades\Validator;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Dompdf\Dompdf;
use Illuminate\Support\Facades\View;
use App\Models\Estoque;
use App\Models\MovimentoEstoque;
use App\Models\InventarioItem;

class ProcessoInventarioController extends Controller
{
    protected $util;

    public function __construct(EstoqueUtil $util)
    {
        $this->util = $util;

        $this->middleware('permission:inventario_view', ['only' => [
            'index', 'criar', 'selecionar', 'manual', 'importacao', 'printListagem', 'gerarModelo'
        ]]);
        $this->middleware('permission:inventario_create', ['only' => [
            'store', 'adicionarProdutos', 'salvarManual', 'uploadExcel'
        ]]);
    }

    /**
     * Lista os inventários ativos
     */
    public function index()
    {
        $inventarios = Inventario::where('empresa_id', request()->empresa_id)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('inventarios.processo.index', compact('inventarios'));
    }

    /**
     * Exibe o formulário para criar um novo inventário
     */
    public function criar()
    {
        $usuarios = User::where('usuario_empresas.empresa_id', request()->empresa_id)
            ->join('usuario_empresas', 'users.id', '=', 'usuario_empresas.usuario_id')
            ->select('users.*')
            ->get();

        return view('inventarios.processo.criar', compact('usuarios'));
    }

    /**
     * Salva um novo inventário (cabeçalho)
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'referencia' => 'required',
            'tipo' => 'required',
        ], [
            'referencia.required' => 'A referência é obrigatória',
            'tipo.required' => 'Selecione o tipo de inventário',
        ]);

        if ($validator->fails()) {
            session()->flash('flash_error', $validator->errors()->first());
            return redirect()->back()->withInput();
        }

        try {
            // Obtém o próximo número sequencial
            $last = Inventario::where('empresa_id', request()->empresa_id)
                ->orderBy('numero_sequencial', 'desc')
                ->where('numero_sequencial', '>', 0)
                ->first();
                
            $numero = $last != null ? $last->numero_sequencial : 0;
            $numero++;
            
            // Cria novo inventário
            $inventario = Inventario::create([
                'numero_sequencial' => $numero,
                'referencia' => $request->referencia,
                'inicio' => $request->inicio ?? date('Y-m-d'),
                'fim' => $request->fim,
                'tipo' => $request->tipo,
                'observacao' => $request->observacao,
                'status' => 1,
                'empresa_id' => request()->empresa_id,
                'usuario_id' => $request->usuario_id ?? get_id_user()
            ]);
            
            __createLog(request()->empresa_id, 'Inventário', 'cadastrar', $request->referencia);
            session()->flash('flash_success', 'Inventário criado com sucesso! Agora selecione os produtos para este inventário.');
            
            return redirect()->route('processo-inventario.selecionar', $inventario->id);
            
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Inventário', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao criar inventário: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Exibe a página de seleção de produtos para o inventário
     */
    public function selecionar($id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        // Produtos já adicionados ao inventário
        $produtosAdicionados = ItemInventario::where('inventario_id', $id)
            ->pluck('produto_id')
            ->toArray();
        
        // Todas as categorias
        $categorias = DB::table('categoria_produtos')
            ->where('empresa_id', request()->empresa_id)
            ->get();
            
        // Iniciar a consulta de produtos
        $query = Produto::where('produtos.empresa_id', request()->empresa_id)
            ->where('produtos.status', 1)
            ->leftJoin('categoria_produtos', 'produtos.categoria_id', '=', 'categoria_produtos.id')
            ->leftJoin('marcas', 'produtos.marca_id', '=', 'marcas.id');
            
        // Aplicar filtros se presentes na requisição
        if (request()->has('busca') && !empty(request()->busca)) {
            $query->where(function($q) {
                $q->where('produtos.nome', 'like', '%' . request()->busca . '%')
                  ->orWhere('produtos.codigo_barras', 'like', '%' . request()->busca . '%');
            });
        }
        
        if (request()->has('categoria_id') && !empty(request()->categoria_id)) {
            $query->where('produtos.categoria_id', request()->categoria_id);
        }
        
        // Aplicar ordenação se especificada
        $orderBy = request()->order_by ?? session('ordem_padrao') ?? 'produtos.nome';
        $orderDir = request()->order_dir ?? session('direcao_ordem') ?? 'asc';
        
        // Validar os campos permitidos para ordenação
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = in_array($orderBy, $allowedOrderFields) ? $orderBy : 'produtos.nome';
        
        // Validar as direções permitidas
        $orderDir = in_array(strtolower($orderDir), ['asc', 'desc']) ? $orderDir : 'asc';
        
        // Completar a consulta com select e ordenação
        $produtos = $query->select(
                'produtos.id',
                'produtos.nome',
                'produtos.codigo_barras',
                'categoria_produtos.nome as categoria_nome',
                'marcas.nome as marca_nome'
            )
            ->orderBy($orderBy, $orderDir)
            ->paginate(20);
            
        // Manter os filtros na paginação
        $produtos->appends(request()->all());
            
        return view('inventarios.processo.selecionar', compact('inventario', 'produtos', 'categorias', 'produtosAdicionados'));
    }

    /**
     * Adiciona produtos ao inventário
     */
    public function adicionarProdutos(Request $request, $id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        if (!$request->has('produtos') || empty($request->produtos)) {
            session()->flash('flash_error', 'Selecione pelo menos um produto para adicionar ao inventário.');
            return redirect()->back();
        }
        
        $produtosExistentes = ItemInventario::where('inventario_id', $id)
            ->pluck('produto_id')
            ->toArray();
            
        $contador = 0;
        
        try {
            DB::beginTransaction();
            
            foreach ($request->produtos as $produtoId) {
                // Verifica se o produto já está no inventário
                if (in_array($produtoId, $produtosExistentes)) {
                    continue;
                }
                
                // Adiciona o produto ao inventário com quantidade 0 (será preenchida posteriormente)
                ItemInventario::create([
                    'inventario_id' => $id,
                    'produto_id' => $produtoId,
                    'quantidade' => 0,
                    'observacao' => '',
                    'usuario_id' => get_id_user()
                ]);
                
                $contador++;
            }
            
            DB::commit();
            
            if ($contador > 0) {
                session()->flash('flash_success', $contador . ' produtos adicionados ao inventário com sucesso!');
            } else {
                session()->flash('flash_info', 'Todos os produtos selecionados já faziam parte do inventário.');
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            __createLog(request()->empresa_id, 'Inventário', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao adicionar produtos: ' . $e->getMessage());
        }
        
        return redirect()->back();
    }

    /**
     * Remove um produto do inventário
     */
    public function removerProduto($inventarioId, $itemId)
    {
        try {
            $item = ItemInventario::findOrFail($itemId);
            
            // Verifica se o item pertence ao inventário informado
            if ($item->inventario_id != $inventarioId) {
                session()->flash('flash_error', 'Item não pertence a este inventário.');
                return redirect()->back();
            }
            
            // Remove o item
            $item->delete();
            
            session()->flash('flash_success', 'Produto removido do inventário com sucesso!');
            
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Inventário', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao remover produto: ' . $e->getMessage());
        }
        
        return redirect()->back();
    }

    /**
     * Exibe a lista de opções de contagem
     */
    public function opcoes($id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        // Verifica se o inventário tem produtos selecionados
        $totalProdutos = ItemInventario::where('inventario_id', $id)->count();
        
        if ($totalProdutos == 0) {
            session()->flash('flash_warning', 'Este inventário não possui produtos selecionados. Adicione produtos antes de prosseguir.');
            return redirect()->route('processo-inventario.selecionar', $id);
        }
        
        return view('inventarios.processo.opcoes', compact('inventario', 'totalProdutos'));
    }

    /**
     * Gera e faz download do PDF com a listagem de produtos para contagem manual
     */
    public function printListagem($id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        // Definir a ordenação padrão
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = session('ordem_padrao') ?? 'produtos.nome';
        $orderDir = session('direcao_ordem') ?? 'asc';
        
        if (!in_array($orderBy, $allowedOrderFields)) {
            $orderBy = 'produtos.nome';
        }
        
        if (!in_array($orderDir, ['asc', 'desc'])) {
            $orderDir = 'asc';
        }
        
        // Produtos selecionados para este inventário com ordenação aplicada
        $itens = ItemInventario::where('inventario_id', $id)
            ->with('produto')
            ->join('produtos', 'item_inventarios.produto_id', '=', 'produtos.id')
            ->orderBy($orderBy, $orderDir)
            ->select('item_inventarios.*')
            ->get();
            
        if ($itens->isEmpty()) {
            session()->flash('flash_warning', 'Este inventário não possui produtos selecionados. Adicione produtos antes de imprimir.');
            return redirect()->route('processo-inventario.selecionar', $id);
        }
        
        // Cria uma instância do Dompdf
        $dompdf = new Dompdf();
        
        // Gera o HTML para o PDF
        $html = View::make('inventarios.processo.pdf.listagem', compact('inventario', 'itens'))->render();
        
        // Carrega o HTML no Dompdf
        $dompdf->loadHtml($html);
        
        // Configura o tamanho do papel e orientação
        $dompdf->setPaper('A4', 'portrait');
        
        // Renderiza o PDF
        $dompdf->render();
        
        // Define o nome do arquivo
        $filename = 'inventario_' . $inventario->numero_sequencial . '_' . date('dmY') . '.pdf';
        
        // Retorna o PDF para download
        return $dompdf->stream($filename, array('Attachment' => true));
    }

    /**
     * Exibe a página de inventário manual com os produtos selecionados
     */
    public function manual(Request $request, $id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        $query = ItemInventario::with(['produto'])
            ->where('inventario_id', $id);
        
        // Verificar se existem produtos no inventário
        if ($query->count() == 0) {
            session()->flash('flash_warning', 'Este inventário não possui produtos selecionados. Adicione produtos antes de prosseguir.');
            return redirect()->route('processo-inventario.selecionar', $id);
        }
        
        // Verificar se há itens selecionados na sessão
        $itens_selecionados = session('itens_selecionados_inventario', []);
        if (!empty($itens_selecionados)) {
            $query->whereIn('item_inventarios.id', $itens_selecionados);
            
            // Remover os itens da sessão depois de usá-los
            session()->forget('itens_selecionados_inventario');
        }
        
        if ($request->has('busca')) {
            $busca = $request->busca;
            $query->whereHas('produto', function ($q) use ($busca) {
                $q->where('nome', 'like', "%{$busca}%")
                  ->orWhere('codigo_barras', 'like', "%{$busca}%");
            });
        }
        
        // Ordenação
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = $request->order_by ?? session('ordem_padrao') ?? 'produtos.nome';
        $orderDir = $request->order_dir ?? session('direcao_ordem') ?? 'asc';
        
        if (!in_array($orderBy, $allowedOrderFields)) {
            $orderBy = 'produtos.nome';
        }
        
        if (!in_array($orderDir, ['asc', 'desc'])) {
            $orderDir = 'asc';
        }
        
        $query->join('produtos', 'item_inventarios.produto_id', '=', 'produtos.id')
              ->orderBy($orderBy, $orderDir)
              ->select('item_inventarios.*'); // Selecionar apenas as colunas da tabela item_inventarios
        
        $itens = $query->paginate(20);
        
        // Manter os filtros na paginação
        $itens->appends(request()->all());
        
        // Verificar se estamos exibindo itens filtrados da tela de comparação
        $vindo_da_comparacao = !empty($itens_selecionados);
        
        return view('inventarios.processo.manual', compact('inventario', 'itens', 'vindo_da_comparacao'));
    }

    /**
     * Atualiza as quantidades dos produtos no inventário manual
     */
    public function salvarManual(Request $request, $id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        try {
            DB::beginTransaction();
            
            $contadorAtualizados = 0;
            
            // Verificar se existem IDs e quantidades enviadas
            if ($request->has('item_id') && $request->has('quantidade')) {
                $itemIds = $request->item_id;
                $quantidades = $request->quantidade;
                $observacoes = $request->observacao ?? [];
                
                // Percorrer os itens e atualizar quantidades
                foreach ($itemIds as $index => $itemId) {
                    // Verificar se existe quantidade informada para este índice
                    if (isset($quantidades[$index]) && $quantidades[$index] !== '') {
                        $item = ItemInventario::findOrFail($itemId);
                        
                        // Atualiza o item
                        $item->quantidade = floatval($quantidades[$index]);
                        $item->observacao = isset($observacoes[$index]) ? $observacoes[$index] : '';
                        $item->save();
                        
                        $contadorAtualizados++;
                    }
                }
            }
            
            DB::commit();
            
            if ($contadorAtualizados > 0) {
                session()->flash('flash_success', $contadorAtualizados . ' produtos atualizados no inventário com sucesso!');
            } else {
                session()->flash('flash_info', 'Nenhuma quantidade foi informada para atualização.');
            }
            
            // Verificar se veio da tela de comparação e redirecionar de volta
            if ($request->has('vindo_da_comparacao') && $request->vindo_da_comparacao) {
                return redirect()->route('processo-inventario.comparar', $id);
            }
            
        } catch (\Exception $e) {
            DB::rollBack();
            __createLog(request()->empresa_id, 'Inventário Manual', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao atualizar quantidades: ' . $e->getMessage());
        }
        
        return redirect()->back();
    }

    /**
     * Exibe a página de importação de inventário por Excel
     */
    public function importacao($id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        // Definir a ordenação padrão
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = session('ordem_padrao') ?? 'produtos.nome';
        $orderDir = session('direcao_ordem') ?? 'asc';
        
        if (!in_array($orderBy, $allowedOrderFields)) {
            $orderBy = 'produtos.nome';
        }
        
        if (!in_array($orderDir, ['asc', 'desc'])) {
            $orderDir = 'asc';
        }
        
        // Produtos selecionados para este inventário com ordenação aplicada
        $itens = ItemInventario::where('inventario_id', $id)
            ->with('produto')
            ->join('produtos', 'item_inventarios.produto_id', '=', 'produtos.id')
            ->orderBy($orderBy, $orderDir)
            ->select('item_inventarios.*')
            ->get();
            
        if ($itens->isEmpty()) {
            session()->flash('flash_warning', 'Este inventário não possui produtos selecionados. Adicione produtos antes de prosseguir.');
            return redirect()->route('processo-inventario.selecionar', $id);
        }
        
        return view('inventarios.processo.importacao', compact('inventario', 'itens'));
    }

    /**
     * Gera e faz download da planilha modelo com os produtos do inventário
     */
    public function gerarModelo($id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        // Definir a ordenação padrão
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = session('ordem_padrao') ?? 'produtos.nome';
        $orderDir = session('direcao_ordem') ?? 'asc';
        
        if (!in_array($orderBy, $allowedOrderFields)) {
            $orderBy = 'produtos.nome';
        }
        
        if (!in_array($orderDir, ['asc', 'desc'])) {
            $orderDir = 'asc';
        }
        
        // Produtos selecionados para este inventário com ordenação aplicada
        $itens = ItemInventario::where('inventario_id', $id)
            ->with('produto')
            ->join('produtos', 'item_inventarios.produto_id', '=', 'produtos.id')
            ->orderBy($orderBy, $orderDir)
            ->select('item_inventarios.*')
            ->get();
            
        if ($itens->isEmpty()) {
            session()->flash('flash_warning', 'Este inventário não possui produtos selecionados. Adicione produtos antes de gerar a planilha.');
            return redirect()->route('processo-inventario.selecionar', $id);
        }
        
        // Cria uma nova planilha
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        
        // Adiciona o título do inventário
        $sheet->setCellValue('A1', 'INVENTÁRIO: ' . $inventario->referencia);
        $sheet->setCellValue('A2', 'DATA: ' . date('d/m/Y'));
        $sheet->setCellValue('A3', 'RESPONSÁVEL: ' . ($inventario->usuario ? $inventario->usuario->name : 'Não definido'));
        
        // Cabeçalhos da planilha
        $sheet->setCellValue('A5', 'ID');
        $sheet->setCellValue('B5', 'Código de Barras');
        $sheet->setCellValue('C5', 'Descrição');
        $sheet->setCellValue('D5', 'Quantidade Contada');
        $sheet->setCellValue('E5', 'Observação');
        
        // Formata o cabeçalho
        $sheet->getStyle('A5:E5')->getFont()->setBold(true);
        
        // Adiciona os dados dos produtos
        $row = 6;
        foreach ($itens as $item) {
            $sheet->setCellValue('A' . $row, $item->produto->id);
            $sheet->setCellValue('B' . $row, $item->produto->codigo_barras);
            $sheet->setCellValue('C' . $row, $item->produto->nome);
            $sheet->setCellValue('D' . $row, '');
            $sheet->setCellValue('E' . $row, '');
            $row++;
        }
        
        // Ajusta largura das colunas
        $sheet->getColumnDimension('A')->setWidth(10);
        $sheet->getColumnDimension('B')->setWidth(20);
        $sheet->getColumnDimension('C')->setWidth(50);
        $sheet->getColumnDimension('D')->setWidth(20);
        $sheet->getColumnDimension('E')->setWidth(40);
        
        // Cria o writer para o arquivo Excel
        $writer = new Xlsx($spreadsheet);
        
        // Define o nome do arquivo
        $filename = 'modelo_inventario_' . $inventario->numero_sequencial . '_' . date('dmY') . '.xlsx';
        
        // Caminho temporário para o arquivo
        $tempPath = storage_path('app/temp/' . $filename);
        
        // Cria o diretório se não existir
        if (!file_exists(dirname($tempPath))) {
            mkdir(dirname($tempPath), 0755, true);
        }
        
        // Salva o arquivo temporariamente
        $writer->save($tempPath);
        
        // Retorna o arquivo para download
        return response()->download($tempPath, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ])->deleteFileAfterSend(true);
    }

    /**
     * Faz o upload e processamento do arquivo Excel
     */
    public function uploadExcel(Request $request, $id)
    {
        $inventario = Inventario::findOrFail($id);
        __validaObjetoEmpresa($inventario);
        
        $validator = Validator::make($request->all(), [
            'arquivo' => 'required|file|mimes:xlsx,xls,csv',
        ], [
            'arquivo.required' => 'Selecione um arquivo para importação',
            'arquivo.file' => 'O arquivo enviado é inválido',
            'arquivo.mimes' => 'O arquivo deve ser do tipo Excel (xlsx, xls) ou CSV',
        ]);

        if ($validator->fails()) {
            session()->flash('flash_error', $validator->errors()->first());
            return redirect()->back()->withInput();
        }

        try {
            // Armazena o arquivo temporariamente
            $file = $request->file('arquivo');
            $path = $file->store('temp');
            $fullPath = Storage::path($path);
            
            // Carrega o arquivo Excel
            $spreadsheet = IOFactory::load($fullPath);
            $worksheet = $spreadsheet->getActiveSheet();
            
            // Obter o mapeamento de ItemInventario IDs para evitar buscas repetidas
            $itensInventario = ItemInventario::where('inventario_id', $id)->get()->keyBy('produto_id');
            
            // Processa os dados
            $sucessos = 0;
            $erros = 0;
            $errosDetalhes = [];
            
            // Determina automaticamente a última linha com dados
            $highestRow = $worksheet->getHighestDataRow();
            
            // Pula os cabeçalhos (assumindo que começam na linha 6)
            for ($row = 6; $row <= $highestRow; $row++) {
                $produtoId = $worksheet->getCell('A' . $row)->getValue();
                $quantidade = $worksheet->getCell('D' . $row)->getValue();
                $observacao = $worksheet->getCell('E' . $row)->getValue() ?? '';
                
                // Verifica se o ID do produto existe e se a quantidade foi informada
                if (!$produtoId || !$quantidade) {
                    continue;
                }
                
                // Verifica se o item existe no inventário
                if (!isset($itensInventario[$produtoId])) {
                    $erros++;
                    $errosDetalhes[] = "Produto com ID {$produtoId} não encontrado neste inventário";
                    continue;
                }
                
                try {
                    // Atualiza o item no inventário
                    $item = $itensInventario[$produtoId];
                    $item->quantidade = __convert_value_bd($quantidade);
                    $item->observacao = $observacao;
                    $item->save();
                    
                    $sucessos++;
                } catch (\Exception $e) {
                    $erros++;
                    $errosDetalhes[] = "Erro ao processar produto ID {$produtoId}: " . $e->getMessage();
                }
            }
            
            // Remove o arquivo temporário
            Storage::delete($path);
            
            if ($sucessos > 0) {
                $mensagem = "{$sucessos} produtos atualizados com sucesso";
                if ($erros > 0) {
                    $mensagem .= ". {$erros} produtos não puderam ser atualizados.";
                }
                session()->flash('flash_success', $mensagem);
                
                // Salva os erros em sessão para exibição detalhada
                if (!empty($errosDetalhes)) {
                    session()->flash('erros_importacao', $errosDetalhes);
                }
                
                // Redireciona para a tela de visualização dos itens importados
                return redirect()->route('processo-inventario.itens-importados', $id);
            } else {
                session()->flash('flash_error', 'Nenhum produto pôde ser atualizado. Verifique os erros.');
                session()->flash('erros_importacao', $errosDetalhes);
                return redirect()->back();
            }
            
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Importação Inventário', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao processar arquivo: ' . $e->getMessage());
        }
        
        return redirect()->back();
    }

    /**
     * Ajusta o estoque dos produtos conforme as quantidades do inventário
     */
    private function ajustarEstoque($inventarioId)
    {
        $itens = ItemInventario::where('inventario_id', $inventarioId)
            ->where('quantidade', '>', 0)  // Apenas itens com quantidade informada
            ->with('produto')
            ->get();
            
        foreach ($itens as $item) {
            if ($item->produto->estoque) {
                // Atualiza o estoque existente
                $estoque = $item->produto->estoque;
                $estoque->quantidade = $item->quantidade;
                $estoque->save();
            } else {
                // Cria um novo registro de estoque
                $this->util->incrementaEstoque($item->produto_id, $item->quantidade, null);
            }
        }
        
        // Atualiza o status do inventário para finalizado (0)
        $inventario = Inventario::find($inventarioId);
        $inventario->status = 0;
        $inventario->fim = date('Y-m-d');
        $inventario->save();
        
        __createLog(request()->empresa_id, 'Inventário', 'ajuste', 'Estoque ajustado pelo inventário #' . $inventarioId);
    }

    /**
     * Salva a configuração de ordem padrão para as listagens
     */
    public function salvarOrdem(Request $request)
    {
        // Validar os campos permitidos
        $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
        $orderBy = $request->ordem_padrao ?? 'produtos.nome';
        $orderDir = $request->direcao_ordem ?? 'asc';
        
        if (!in_array($orderBy, $allowedOrderFields)) {
            $orderBy = 'produtos.nome';
        }
        
        if (!in_array($orderDir, ['asc', 'desc'])) {
            $orderDir = 'asc';
        }
        
        // Salvar a configuração na sessão
        session(['ordem_padrao' => $orderBy]);
        session(['direcao_ordem' => $orderDir]);
        
        // Registrar a operação no log
        __createLog(request()->empresa_id, 'Inventário', 'configuração', 'Ordem padrão das listagens atualizada para ' . $orderBy . ' ' . $orderDir);
        
        session()->flash('flash_success', 'Configuração de ordenação atualizada com sucesso!');
        return redirect()->route('processo-inventario.index');
    }

    /**
     * Exibe a tela de comparação entre o estoque atual e as quantidades contadas
     */
    public function compararContagem($id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            
            // Obtém todos os itens do inventário com seus produtos e estoques
            $itens = ItemInventario::with(['produto', 'produto.estoque'])
                ->where('inventario_id', $id)
                ->get();
            
            $analise = [];
            $total_itens = $itens->count();
            $itens_contados = 0;
            $divergencias = 0;
            
            foreach ($itens as $item) {
                // Ignora produtos sem estoque cadastrado
                if (!$item->produto->estoque) {
                    continue;
                }
                
                $estoque_atual = $item->produto->estoque->quantidade;
                $quantidade_contada = $item->quantidade ?: 0;
                
                // Calcula divergência
                $divergencia = $quantidade_contada - $estoque_atual;
                
                // Calcula percentual de diferença
                $percentual = 0;
                if ($estoque_atual > 0) {
                    $percentual = round(($divergencia / $estoque_atual) * 100, 2);
                } elseif ($quantidade_contada > 0) {
                    $percentual = 100; // Se o estoque atual é zero mas foi contado algo
                }
                
                // Contabiliza para o resumo
                if ($quantidade_contada > 0) {
                    $itens_contados++;
                }
                
                if ($divergencia != 0) {
                    $divergencias++;
                }
                
                $analise[] = [
                    'item_id' => $item->id,
                    'produto_id' => $item->produto->id,
                    'produto_nome' => $item->produto->nome,
                    'codigo_barras' => $item->produto->codigo_barras,
                    'estoque_atual' => $estoque_atual,
                    'quantidade_contada' => $quantidade_contada,
                    'divergencia' => $divergencia,
                    'percentual' => $percentual,
                    'observacao' => $item->observacao
                ];
            }
            
            // Ordena a análise por maior divergência (em valor absoluto)
            usort($analise, function($a, $b) {
                return abs($b['divergencia']) <=> abs($a['divergencia']);
            });
            
            // Calcula percentual de divergência geral
            $percentual_divergencia = 0;
            if ($itens_contados > 0) {
                $percentual_divergencia = round(($divergencias / $itens_contados) * 100, 2);
            }
            
            $resumo = [
                'total_itens' => $total_itens,
                'itens_contados' => $itens_contados,
                'divergencias' => $divergencias,
                'percentual_divergencia' => $percentual_divergencia
            ];
            
            return view('inventarios.processo.comparar', compact('inventario', 'analise', 'resumo'));
        } catch (\Exception $e) {
            return redirect()->route('processo-inventario.index')
                ->with('flash_error', 'Erro ao comparar contagem: ' . $e->getMessage());
        }
    }
    
    /**
     * Processa o ajuste de estoque dos itens selecionados
     */
    public function ajustarSelecionados(Request $request, $id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            $itens_selecionados = $request->input('itens_selecionados', []);
            
            if (empty($itens_selecionados)) {
                return redirect()->route('processo-inventario.comparar', $id)
                    ->with('flash_warning', 'Nenhum item foi selecionado para ajuste.');
            }
            
            // Armazenar os IDs dos itens selecionados na sessão
            session(['itens_selecionados_inventario' => $itens_selecionados]);
            
            // Redirecionar para a tela manual com os itens selecionados
            return redirect()->route('processo-inventario.manual', $id);
            
        } catch (\Exception $e) {
            return redirect()->route('processo-inventario.comparar', $id)
                ->with('flash_error', 'Erro ao processar itens selecionados: ' . $e->getMessage());
        }
    }

    /**
     * Exibe a tela de confirmação para finalizar o inventário
     */
    public function confirmarFinalizacao($id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            __validaObjetoEmpresa($inventario);
            
            // Verificar se o inventário já está finalizado
            if ($inventario->status == 0 || $inventario->status == 'finalizado') {
                return redirect()->route('processo-inventario.index')
                    ->with('flash_warning', 'Este inventário já foi finalizado.');
            }
            
            // Verificar se existem produtos com contagem registrada
            $itensContados = ItemInventario::where('inventario_id', $id)
                ->where('quantidade', '>', 0)
                ->count();
                
            if ($itensContados == 0) {
                return redirect()->route('processo-inventario.comparar', $id)
                    ->with('flash_warning', 'Não há produtos com contagem registrada para finalizar o inventário.');
            }
            
            // Obtém todos os itens do inventário com seus produtos e estoques para exibir na tela de confirmação
            $itens = ItemInventario::with(['produto', 'produto.estoque'])
                ->where('inventario_id', $id)
                ->where('quantidade', '>', 0)
                ->get();
            
            return view('inventarios.processo.confirmar-finalizacao', compact('inventario', 'itens', 'itensContados'));
            
        } catch (\Exception $e) {
            return redirect()->route('processo-inventario.index')
                ->with('flash_error', 'Erro ao preparar finalização do inventário: ' . $e->getMessage());
        }
    }
    
    /**
     * Processa a finalização do inventário, zerando e depois atualizando o estoque
     */
    public function finalizarInventario(Request $request, $id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            __validaObjetoEmpresa($inventario);
            
            // Verificar se o inventário já está finalizado
            if ($inventario->status == 0 || $inventario->status == 'finalizado') {
                return redirect()->route('processo-inventario.index')
                    ->with('flash_warning', 'Este inventário já foi finalizado.');
            }
            
            // Obter todos os itens com contagem registrada
            $itens = ItemInventario::with(['produto', 'produto.estoque'])
                ->where('inventario_id', $id)
                ->where('quantidade', '>', 0)
                ->get();
                
            if ($itens->isEmpty()) {
                return redirect()->route('processo-inventario.comparar', $id)
                    ->with('flash_warning', 'Não há produtos com contagem registrada para finalizar o inventário.');
            }
            
            // Array para armazenar o log do processo
            $log_processo = [
                'inicio' => now()->format('d/m/Y H:i:s'),
                'zeragem' => [],
                'atualizacao' => [],
                'erros' => []
            ];
            
            DB::beginTransaction();
            
            try {
                // Etapa 1: Zerar o estoque de todos os produtos no inventário
                foreach ($itens as $item) {
                    try {
                        if (!$item->produto->estoque) {
                            // Cria um registro de estoque se não existir
                            $estoque = new Estoque();
                            $estoque->produto_id = $item->produto->id;
                            $estoque->quantidade = 0;
                            $estoque->save();
                            
                            // Recarrega o relacionamento
                            $item->produto->load('estoque');
                        }
                        
                        // Registra o valor anterior
                        $quantidade_anterior = $item->produto->estoque->quantidade;
                        
                        // Zera o estoque
                        $item->produto->estoque->quantidade = 0;
                        $item->produto->estoque->save();
                        
                        // Registra o movimento de estoque (zeragem)
                        $movimento = new MovimentoEstoque();
                        $movimento->produto_id = $item->produto->id;
                        $movimento->quantidade_anterior = $quantidade_anterior;
                        $movimento->quantidade_nova = 0;
                        $movimento->tipo = 'ajuste';
                        $movimento->operacao = 'saida';
                        $movimento->diferenca = -$quantidade_anterior;
                        $movimento->observacao = "Zeragem para inventário #{$inventario->numero_sequencial}";
                        $movimento->user_id = auth()->id();
                        $movimento->inventario_id = $inventario->id;
                        $movimento->save();
                        
                        $log_processo['zeragem'][] = [
                            'produto_id' => $item->produto->id,
                            'produto_nome' => $item->produto->nome,
                            'quantidade_anterior' => $quantidade_anterior,
                            'status' => 'sucesso'
                        ];
                    } catch (\Exception $e) {
                        $log_processo['zeragem'][] = [
                            'produto_id' => $item->produto->id,
                            'produto_nome' => $item->produto->nome,
                            'erro' => $e->getMessage(),
                            'status' => 'erro'
                        ];
                        $log_processo['erros'][] = "Erro ao zerar estoque do produto {$item->produto->nome}: {$e->getMessage()}";
                        throw $e;
                    }
                }
                
                // Etapa 2: Atualizar com as quantidades contadas
                foreach ($itens as $item) {
                    try {
                        // Atualiza o estoque com a quantidade contada
                        $item->produto->estoque->quantidade = $item->quantidade;
                        $item->produto->estoque->save();
                        
                        // Registra o movimento de estoque (atualização)
                        $movimento = new MovimentoEstoque();
                        $movimento->produto_id = $item->produto->id;
                        $movimento->quantidade_anterior = 0;
                        $movimento->quantidade_nova = $item->quantidade;
                        $movimento->tipo = 'ajuste';
                        $movimento->operacao = 'entrada';
                        $movimento->diferenca = $item->quantidade;
                        $movimento->observacao = "Atualização por inventário #{$inventario->numero_sequencial}";
                        $movimento->user_id = auth()->id();
                        $movimento->inventario_id = $inventario->id;
                        $movimento->save();
                        
                        $log_processo['atualizacao'][] = [
                            'produto_id' => $item->produto->id,
                            'produto_nome' => $item->produto->nome,
                            'quantidade_nova' => $item->quantidade,
                            'status' => 'sucesso'
                        ];
                    } catch (\Exception $e) {
                        $log_processo['atualizacao'][] = [
                            'produto_id' => $item->produto->id,
                            'produto_nome' => $item->produto->nome,
                            'erro' => $e->getMessage(),
                            'status' => 'erro'
                        ];
                        $log_processo['erros'][] = "Erro ao atualizar estoque do produto {$item->produto->nome}: {$e->getMessage()}";
                        throw $e;
                    }
                }
                
                // Finaliza o inventário
                $inventario->status = 'finalizado';
                $inventario->fim = now();
                $inventario->data_ajuste = now();
                $inventario->save();
                
                $log_processo['fim'] = now()->format('d/m/Y H:i:s');
                $log_processo['total_produtos'] = count($itens);
                $log_processo['status_final'] = 'concluido';
                
                // Salva o log completo do processo em uma sessão para exibição na tela
                session(['log_processo_inventario' => $log_processo]);
                
                DB::commit();
                
                // Registra no log do sistema
                __createLog(request()->empresa_id, 'Inventário', 'finalizar', "Inventário #{$inventario->numero_sequencial} finalizado com sucesso. Total de produtos: " . count($itens));
                
                return redirect()->route('processo-inventario.resultado-finalizacao', $inventario->id)
                    ->with('flash_success', "Inventário finalizado com sucesso! O estoque de " . count($itens) . " produtos foi atualizado.");
                
            } catch (\Exception $e) {
                DB::rollBack();
                
                $log_processo['status_final'] = 'erro';
                $log_processo['erro_fatal'] = $e->getMessage();
                $log_processo['fim'] = now()->format('d/m/Y H:i:s');
                
                // Salva o log do processo mesmo em caso de erro
                session(['log_processo_inventario' => $log_processo]);
                
                __createLog(request()->empresa_id, 'Inventário', 'erro', "Erro ao finalizar inventário #{$inventario->numero_sequencial}: " . $e->getMessage());
                
                return redirect()->route('processo-inventario.resultado-finalizacao', $inventario->id)
                    ->with('flash_error', 'Erro ao finalizar inventário: ' . $e->getMessage());
            }
            
        } catch (\Exception $e) {
            return redirect()->route('processo-inventario.index')
                ->with('flash_error', 'Erro ao finalizar inventário: ' . $e->getMessage());
        }
    }
    
    /**
     * Exibe o resultado da finalização do inventário
     */
    public function resultadoFinalizacao($id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            __validaObjetoEmpresa($inventario);
            
            $log_processo = session('log_processo_inventario', null);
            
            if (!$log_processo) {
                return redirect()->route('processo-inventario.index')
                    ->with('flash_warning', 'Não há informações de processamento para este inventário.');
            }
            
            return view('inventarios.processo.resultado-finalizacao', compact('inventario', 'log_processo'));
            
        } catch (\Exception $e) {
            return redirect()->route('processo-inventario.index')
                ->with('flash_error', 'Erro ao exibir resultado da finalização: ' . $e->getMessage());
        }
    }

    /**
     * Gera e faz download do PDF com o relatório completo do inventário
     */
    public function printRelatorio($id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            __validaObjetoEmpresa($inventario);
            
            // Obtém todos os itens do inventário com seus produtos e estoques
            $itens = ItemInventario::with(['produto', 'produto.estoque'])
                ->where('inventario_id', $id)
                ->get();
            
            if ($itens->isEmpty()) {
                session()->flash('flash_warning', 'Este inventário não possui produtos para gerar relatório.');
                return redirect()->route('processo-inventario.index');
            }
            
            $analise = [];
            $divergentes = [];
            $total_itens = $itens->count();
            $itens_contados = 0;
            $divergencias = 0;
            
            foreach ($itens as $item) {
                // Se o produto não tem estoque, considera o estoque atual como zero
                $estoque_atual = 0;
                if ($item->produto->estoque) {
                    $estoque_atual = $item->produto->estoque->quantidade;
                }
                
                $quantidade_contada = $item->quantidade ?: 0;
                
                // Calcula divergência
                $divergencia = $quantidade_contada - $estoque_atual;
                
                // Calcula percentual de diferença
                $percentual = 0;
                if ($estoque_atual > 0) {
                    $percentual = round(($divergencia / $estoque_atual) * 100, 2);
                } elseif ($quantidade_contada > 0) {
                    $percentual = 100; // Se o estoque atual é zero mas foi contado algo
                }
                
                // Contabiliza para o resumo
                if ($quantidade_contada > 0) {
                    $itens_contados++;
                }
                
                // Dados do item para o relatório
                $item_analise = [
                    'item_id' => $item->id,
                    'produto_id' => $item->produto->id,
                    'produto_nome' => $item->produto->nome,
                    'codigo_barras' => $item->produto->codigo_barras,
                    'estoque_atual' => $estoque_atual,
                    'quantidade_contada' => $quantidade_contada,
                    'divergencia' => $divergencia,
                    'percentual' => $percentual,
                    'observacao' => $item->observacao
                ];
                
                $analise[] = $item_analise;
                
                // Se há divergência, adiciona à lista de divergentes
                if ($divergencia != 0) {
                    $divergencias++;
                    $divergentes[] = $item_analise;
                }
            }
            
            // Calcular percentual geral de divergência
            $percentual_divergencia = 0;
            if ($itens_contados > 0) {
                $percentual_divergencia = round(($divergencias / $itens_contados) * 100, 2);
            }
            
            // Resumo para o relatório
            $resumo = [
                'total_itens' => $total_itens,
                'itens_contados' => $itens_contados,
                'divergencias' => $divergencias,
                'percentual_divergencia' => $percentual_divergencia
            ];
            
            // Ordenar divergentes por maior divergência
            usort($divergentes, function($a, $b) {
                return abs($b['divergencia']) <=> abs($a['divergencia']);
            });
            
            // Cria uma instância do Dompdf
            $dompdf = new Dompdf();
            
            // Gera o HTML para o PDF usando o template de relatório
            $html = View::make('inventarios.processo.pdf.relatorio', compact('inventario', 'analise', 'divergentes', 'resumo'))->render();
            
            // Carrega o HTML no Dompdf
            $dompdf->loadHtml($html);
            
            // Configura o tamanho do papel e orientação
            $dompdf->setPaper('A4', 'portrait');
            
            // Renderiza o PDF
            $dompdf->render();
            
            // Define o nome do arquivo
            $filename = 'relatorio_inventario_' . $inventario->numero_sequencial . '_' . date('dmY') . '.pdf';
            
            // Retorna o PDF para download
            return $dompdf->stream($filename, array('Attachment' => true));
            
        } catch (\Exception $e) {
            session()->flash('flash_error', 'Erro ao gerar relatório: ' . $e->getMessage());
            return redirect()->route('processo-inventario.index');
        }
    }
    
    /**
     * Exibe o grid com os itens importados após a importação
     */
    public function itensImportados($id)
    {
        try {
            $inventario = Inventario::findOrFail($id);
            __validaObjetoEmpresa($inventario);
            
            // Define a ordenação padrão
            $allowedOrderFields = ['produtos.id', 'produtos.nome', 'produtos.codigo_barras'];
            $orderBy = session('ordem_padrao') ?? 'produtos.nome';
            $orderDir = session('direcao_ordem') ?? 'asc';
            
            if (!in_array($orderBy, $allowedOrderFields)) {
                $orderBy = 'produtos.nome';
            }
            
            if (!in_array($orderDir, ['asc', 'desc'])) {
                $orderDir = 'asc';
            }
            
            // Busca apenas itens com quantidade informada (itens que foram importados)
            $itens = ItemInventario::where('inventario_id', $id)
                ->where('quantidade', '>', 0)
                ->with(['produto', 'produto.estoque'])
                ->join('produtos', 'item_inventarios.produto_id', '=', 'produtos.id')
                ->orderBy($orderBy, $orderDir)
                ->select('item_inventarios.*')
                ->get();
            
            $total = $itens->count();
            
            return view('inventarios.processo.itens-importados', compact('inventario', 'itens', 'total'));
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Inventário', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Erro ao visualizar itens importados: ' . $e->getMessage());
            return redirect()->route('processo-inventario.importacao', $id);
        }
    }
} 