<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\ContaPagar;
use App\Models\ContaReceber;
use App\Models\Cte;
use App\Models\Mdfe;
use Illuminate\Http\Request;
use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\Cliente;
use App\Models\Produto;
use App\Models\Localizacao;
use Carbon\Carbon;

class GraficoController extends Controller
{
    public function dadosDards(Request $request){
        $periodo = $request->periodo;
        $empresa_id = $request->empresa_id;
        $usuario_id = $request->usuario_id;
        $local_id = $request->local_id;
        
        // Adicionar log para debug do período recebido
        \Illuminate\Support\Facades\Log::info('Período recebido: ' . $periodo);

        $locais = Localizacao::where('usuario_localizacaos.usuario_id', $usuario_id)
        ->select('localizacaos.*')
        ->join('usuario_localizacaos', 'usuario_localizacaos.localizacao_id', '=', 'localizacaos.id')
        ->where('localizacaos.status', 1)->get();
        $locais = $locais->pluck(['id']);

        // Configurar filtros de data baseados no período
        $startDate = null;
        $endDate = null;
        
        switch($periodo) {
            case 'today':
                $startDate = Carbon::today();
                $endDate = Carbon::today();
                break;
            case 'week':
                $startDate = Carbon::now()->startOfWeek();
                $endDate = Carbon::now()->endOfWeek();
                break;
            case 'month':
                $startDate = Carbon::now()->startOfMonth();
                $endDate = Carbon::now()->endOfMonth();
                break;
            case 'year':
                $startDate = Carbon::now()->startOfYear();
                $endDate = Carbon::now()->endOfYear();
                break;
            default:
                // Verificar se é uma data personalizada (formato: dd/mm/yyyy - dd/mm/yyyy)
                if (strpos($periodo, '/') !== false) {
                    try {
                        // Se for um intervalo (contém " - ")
                        if (strpos($periodo, ' - ') !== false) {
                            list($start, $end) = explode(' - ', $periodo);
                            $startDate = Carbon::createFromFormat('d/m/Y', trim($start));
                            $endDate = Carbon::createFromFormat('d/m/Y', trim($end));
                        } 
                        // Se for uma única data
                        else {
                            $startDate = Carbon::createFromFormat('d/m/Y', $periodo);
                            $endDate = $startDate->copy();
                        }
                    } catch (\Exception $e) {
                        // Em caso de erro no formato, usar mês atual
                        \Illuminate\Support\Facades\Log::error('Erro ao processar data personalizada: ' . $e->getMessage());
                        $startDate = Carbon::now()->startOfMonth();
                        $endDate = Carbon::now()->endOfMonth();
                    }
                } else {
                    // Padrão: mês atual
                    $startDate = Carbon::now()->startOfMonth();
                    $endDate = Carbon::now()->endOfMonth();
                }
        }
        
        \Illuminate\Support\Facades\Log::info('Período processado: ' . $startDate->format('Y-m-d') . ' até ' . $endDate->format('Y-m-d'));

        // Valores padrão (todos os dados)
        $somaVendas = Nfe::where('empresa_id', $empresa_id)
        ->where('tpNF', 1)
        ->where('orcamento', 0)
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');

        $somaVendasPdv = Nfce::where('empresa_id', $empresa_id)
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');

        // Valores específicos do período
        $somaVendasPeriodo = Nfe::where('empresa_id', $empresa_id)
        ->where('tpNF', 1)
        ->where('orcamento', 0)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');

        $somaVendasPdvPeriodo = Nfce::where('empresa_id', $empresa_id)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');

        $totalClientes = Cliente::where('empresa_id', $empresa_id)->count('id');
        
        $totalClientesPeriodo = Cliente::where('empresa_id', $empresa_id)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->count('id');

        $totalProdutos = Produto::where('empresa_id', $empresa_id)
        ->select('produtos.*')
        ->when($local_id, function ($query) use ($local_id) {
            return $query->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->where('produto_localizacaos.localizacao_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->whereIn('produto_localizacaos.localizacao_id', $locais);
        })
        ->count('produtos.id');
        
        $totalProdutosPeriodo = Produto::where('empresa_id', $empresa_id)
        ->select('produtos.*')
        ->whereBetween('produtos.created_at', [$startDate, $endDate])
        ->when($local_id, function ($query) use ($local_id) {
            return $query->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->where('produto_localizacaos.localizacao_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->join('produto_localizacaos', 'produto_localizacaos.produto_id', '=', 'produtos.id')
            ->whereIn('produto_localizacaos.localizacao_id', $locais);
        })
        ->count('produtos.id');

        $somaCompras = Nfe::where('empresa_id', $empresa_id)
        ->where('tpNF', 0)
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');
        
        $somaComprasPeriodo = Nfe::where('empresa_id', $empresa_id)
        ->where('tpNF', 0)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('total');

        // Soma contas a receber (pendentes e parciais, apenas valor pendente)
        $somaContaReceber = ContaReceber::where('empresa_id', $empresa_id)
            ->whereIn('status', [0, 2])
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get()
            ->sum(function($conta) {
                return $conta->valorPendente();
            });
        
        $somaContaReceberPeriodo = ContaReceber::where('empresa_id', $empresa_id)
            ->whereIn('status', [0, 2])
            ->whereBetween('data_vencimento', [$startDate, $endDate])
            ->when($local_id, function ($query) use ($local_id) {
                return $query->where('local_id', $local_id);
            })
            ->when(!$local_id, function ($query) use ($locais) {
                return $query->whereIn('local_id', $locais);
            })
            ->get()
            ->sum(function($conta) {
                return $conta->valorPendente();
            });

        $somaContaPagar = ContaPagar::where('empresa_id', $empresa_id)
        ->where('status', 0)
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('valor_integral');
        
        $somaContaPagarPeriodo = ContaPagar::where('empresa_id', $empresa_id)
        ->where('status', 0)
        ->whereBetween('data_vencimento', [$startDate, $endDate])
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->sum('valor_integral');

        // Dados de NFes para estatísticas
        $totalNfeCount = Nfe::where('empresa_id', $empresa_id)
        ->where(function($q) {
            $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
        })
        ->where('tpNF', 1)
        ->count('id');
        
        $totalNfeCountPeriodo = Nfe::where('empresa_id', $empresa_id)
        ->where(function($q) {
            $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
        })
        ->where('tpNF', 1)
        ->whereBetween('created_at', [$startDate, $endDate])
        ->count('id');

        $totalNfceCount = Nfce::where('empresa_id', $empresa_id)
        ->where(function($q) {
            $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
        })
        ->count('id');
        
        $totalNfceCountPeriodo = Nfce::where('empresa_id', $empresa_id)
        ->where(function($q) {
            $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
        })
        ->whereBetween('created_at', [$startDate, $endDate])
        ->count('id');

        $totalVendasMes = $somaVendas + $somaVendasPdv;
        $totalVendasMesPeriodo = $somaVendasPeriodo + $somaVendasPdvPeriodo;
        
        $totalComprasMes = $somaCompras;
        $totalComprasMesPeriodo = $somaComprasPeriodo;

        $data = [
            // Valores padrão
            'vendas' => $somaVendas + $somaVendasPdv,
            'compras' => $somaCompras,
            'clientes' => $totalClientes,
            'produtos' => $totalProdutos,
            'contas_receber' => $somaContaReceber,
            'contas_pagar' => $somaContaPagar,
            'totalNfeCount' => $totalNfeCount,
            'totalNfceCount' => $totalNfceCount,
            'totalVendasMes' => $totalVendasMes,
            'totalComprasMes' => $totalComprasMes,
            
            // Valores do período selecionado
            'vendas_periodo' => $somaVendasPeriodo + $somaVendasPdvPeriodo,
            'compras_periodo' => $somaComprasPeriodo,
            'clientes_periodo' => $totalClientesPeriodo,
            'produtos_periodo' => $totalProdutosPeriodo,
            'contas_receber_periodo' => $somaContaReceberPeriodo,
            'contas_pagar_periodo' => $somaContaPagarPeriodo,
            'totalNfeCount_periodo' => $totalNfeCountPeriodo,
            'totalNfceCount_periodo' => $totalNfceCountPeriodo,
            'totalVendasMes_periodo' => $totalVendasMesPeriodo,
            'totalComprasMes_periodo' => $totalComprasMesPeriodo,
            
            // Informações do período para debugging
            'periodo' => $periodo,
            'data_inicio' => $startDate->format('Y-m-d'),
            'data_fim' => $endDate->format('Y-m-d')
        ];

        return response()->json($data, 200);
    }

    public function graficoVendasMes(Request $request)
    {

        $diaHoje = date('d');
        $mes = date('m');
        $data = [];
        for ($i = 1; $i <= $diaHoje; $i++) {
            $totalNfe = Nfe::where('empresa_id', $request->empresa_id)
            ->where('estado', '!=', 'cancelado')
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->where('tpNF', 1)
            ->sum('total');

            $totalNfce = Nfce::where('empresa_id', $request->empresa_id)
            ->where('estado', '!=', 'cancelado')
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->sum('total');

            array_push($data, [
                'dia' => ($i < 10 ? "0$i" : $i) . "/$mes",
                'valor' => $totalNfe + $totalNfce
            ]);
        }
        return response()->json($data, 200);
    }

    public function graficoComprasMes(Request $request)
    {

        $diaHoje = date('d');
        $mes = date('m');
        $data = [];
        for ($i = 1; $i <= $diaHoje; $i++) {
            $totalNfe = Nfe::where('empresa_id', $request->empresa_id)
            ->where('estado', '!=', 'cancelado')
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->where('tpNF', 0)
            ->sum('total');

            array_push($data, [
                'dia' => ($i < 10 ? "0$i" : $i) . "/$mes",
                'valor' => $totalNfe
            ]);
        }
        return response()->json($data, 200);
    }

    public function graficoMes(Request $request)
    {
        $diaHoje = date('d');
        $mes = date('m');
        $data = [];
        for ($i = 1; $i <= $diaHoje; $i++) {
            $totalNfe = Nfe::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->sum('total');

            $totalNfce = Nfce::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->sum('total');

            array_push($data, [
                'dia' => ($i < 10 ? "0$i" : $i) . "/$mes",
                'valor' => $totalNfe + $totalNfce
            ]);
        }
        return response()->json($data, 200);
    }

    public function graficoMesContador(Request $request)
    {
        $diaHoje = date('d');
        $mes = date('m');
        $data = [];
        for ($i = 1; $i <= $diaHoje; $i++) {
            $totalNfe = Nfe::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->count('id');

            $totalNfce = Nfce::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', date('m'))
            ->whereDay('created_at', ($i < 10 ? "0$i" : $i))
            ->count('id');

            array_push($data, [
                'dia' => ($i < 10 ? "0$i" : $i) . "/$mes",
                'valor' => $totalNfe + $totalNfce
            ]);
        }
        return response()->json($data, 200);
    }

    public function graficoUltMeses(Request $request)
    {
        $mes = (int)date('m');
        $ano = date('Y');
        $data = [];
        for ($i = 0; $i < 4; $i++) {
            $totalNfe = Nfe::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('total');

            $totalNfce = Nfce::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('total');

            array_push($data, [
                'dia' => $this->getMes($mes - 1) . "/$ano",
                'valor' => $totalNfe + $totalNfce
            ]);

            if ($mes == 1) {
                $mes = 12;
                $ano--;
            } else {
                $mes--;
            }
        }
        return response()->json($data, 200);
    }

    private function getMes($m)
    {
        $meses = [
            'Jan', 'Fev', 'Mar', 'Abr', 'Mai', 'Jun', 'Jul', 'Ago', 'Set',
            'Out', 'Nov', 'Dez'
        ];
        return $meses[$m];
    }

    public function graficoContaReceber(Request $request)
    {
        $mes = (int)date('m');
        $ano = date('Y');
        $data = [];
        for ($i = 0; $i < 4; $i++) {
            $receber = ContaReceber::where('empresa_id', $request->empresa_id)
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            $pendente = ContaReceber::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('status', false);
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            $quitado = ContaReceber::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('status', true);
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            array_push($data, [
                'dia' => $this->getMes($mes - 1) . "/$ano",
                'valor' => $receber,
                'valorPendente' => $pendente,
                'valorQuitado' => $quitado
            ]);

            if ($mes == 1) {
                $mes = 12;
                $ano--;
            } else {
                $mes--;
            }
        }
        return response()->json($data, 200);
    }

    public function graficoContaPagar(Request $request)
    {
        $mes = (int)date('m');
        $ano = date('Y');
        $data = [];
        for ($i = 0; $i < 4; $i++) {
            $pagar = ContaPagar::where('empresa_id', $request->empresa_id)
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            $pendentes = ContaPagar::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('status', false);
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            $quitadas = ContaPagar::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('status', true);
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->sum('valor_integral');

            array_push($data, [
                'dia' => $this->getMes($mes - 1) . "/$ano",
                'valor' => $pagar,
                'valorPendente' => $pendentes,
                'valorQuitado' => $quitadas
            ]);

            if ($mes == 1) {
                $mes = 12;
                $ano--;
            } else {
                $mes--;
            }
        }
        return response()->json($data, 200);
    }

    public function graficoMesCte(Request $request)
    {
        $mes = (int)date('m');
        $ano = date('Y');
        $data = [];
        for ($i = 0; $i < 4; $i++) {
            $totalNfe = Cte::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado', 'aprovado')->orWhere('estado', 'cancelado');
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->count('id');

            array_push($data, [
                'dia' => $this->getMes($mes - 1) . "/$ano",
                'valor' => $totalNfe
            ]);

            if ($mes == 1) {
                $mes = 12;
                $ano--;
            } else {
                $mes--;
            }
        }
        return response()->json($data, 200);
    }

    public function graficoMesMdfe(Request $request)
    {
        $mes = (int)date('m');
        $ano = date('Y');
        $data = [];
        for ($i = 0; $i < 4; $i++) {
            $totalNfe = Mdfe::where('empresa_id', $request->empresa_id)
            ->where(function ($q) {
                $q->where('estado_emissao', 'aprovado')->orWhere('estado_emissao', 'cancelado');
            })
            ->whereMonth('created_at', $mes)
            ->whereYear('created_at', $ano)
            ->count('id');

            array_push($data, [
                'dia' => $this->getMes($mes - 1) . "/$ano",
                'valor' => $totalNfe
            ]);

            if ($mes == 1) {
                $mes = 12;
                $ano--;
            } else {
                $mes--;
            }
        }
        return response()->json($data, 200);
    }

}
