<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Mail\EmpresasInativasNotificacao;
use App\Services\NotificacaoService;
use App\Models\Empresa;
use App\Models\Nfe;
use App\Models\Nfce;
use App\Models\User;
use App\Models\AcaoLog;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class NotificarEmpresasInativas extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notificacoes:empresas-inativas';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Envia notificação sobre empresas sem atividade há mais de 30 dias';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->info('Buscando empresas inativas...');
        
        // Data limite (30 dias atrás)
        $dataLimite = Carbon::now()->subDays(30);
        
        // Buscar empresas inativas
        $empresasInativas = $this->buscarEmpresasInativas($dataLimite);
        
        if (count($empresasInativas) == 0) {
            $this->info('Nenhuma empresa inativa encontrada.');
            return 0;
        }
        
        $this->info('Encontradas ' . count($empresasInativas) . ' empresas inativas.');
        
        // Obtém lista de destinatários
        $destinatarios = NotificacaoService::getDestinatarios();
        
        try {
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new EmpresasInativasNotificacao($empresasInativas, $dataLimite));
                $this->info("Email enviado com sucesso para {$dest}");
                Log::info("Notificação de empresas inativas enviada com sucesso para {$dest}");
            }
            
            $this->info('Notificação de empresas inativas enviada com sucesso!');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('Erro ao enviar notificação de empresas inativas: ' . $e->getMessage());
            Log::error('Erro ao enviar notificação de empresas inativas', [
                'erro' => $e->getMessage()
            ]);
            return 1;
        }
    }
    
    /**
     * Busca empresas que não tiveram atividade desde a data limite
     * 
     * @param Carbon $dataLimite
     * @return array
     */
    private function buscarEmpresasInativas($dataLimite)
    {
        // Todas as empresas
        $todasEmpresas = Empresa::select('id', 'nome', 'razao_social', 'cpf_cnpj', 'created_at')->get();
        $empresasInativas = [];
        
        foreach ($todasEmpresas as $empresa) {
            // Verifica NFes recentes
            $nfesRecentes = Nfe::where('empresa_id', $empresa->id)
                ->where('created_at', '>=', $dataLimite)
                ->count();
                
            if ($nfesRecentes > 0) {
                continue; // Empresa com NFes recentes, não é inativa
            }
            
            // Verifica NFCes recentes
            $nfcesRecentes = Nfce::where('empresa_id', $empresa->id)
                ->where('created_at', '>=', $dataLimite)
                ->count();
                
            if ($nfcesRecentes > 0) {
                continue; // Empresa com NFCes recentes, não é inativa
            }
            
            // Verifica logs de ação recentes
            $logsRecentes = AcaoLog::where('empresa_id', $empresa->id)
                ->where('created_at', '>=', $dataLimite)
                ->count();
                
            if ($logsRecentes > 0) {
                continue; // Empresa com logs recentes, não é inativa
            }
            
            // Verifica login de usuários recentes
            $usuariosRecentes = User::join('usuario_empresas', 'users.id', '=', 'usuario_empresas.usuario_id')
                ->where('usuario_empresas.empresa_id', $empresa->id)
                ->where('users.last_login_at', '>=', $dataLimite)
                ->count();
                
            if ($usuariosRecentes > 0) {
                continue; // Empresa com logins recentes, não é inativa
            }
            
            // Se chegou até aqui, a empresa é inativa
            $empresasInativas[] = [
                'id' => $empresa->id,
                'nome' => $empresa->nome,
                'razao_social' => $empresa->razao_social,
                'cpf_cnpj' => $empresa->cpf_cnpj,
                'created_at' => $empresa->created_at,
                'dias_inativa' => $dataLimite->diffInDays($empresa->updated_at)
            ];
        }
        
        return $empresasInativas;
    }
} 