<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Empresa;
use App\Models\NotificacaoCobranca;
use App\Models\NotificacaoCobrancaConfig;
use App\Services\EvolutionAPIService;
use App\Services\OnzFinanceService;
use App\Utils\UploadUtil;
use Illuminate\Support\Facades\Http;

class NotificacaoCobrancaController extends Controller
{
    public $evolutionApiService;
    protected $util;

    public function __construct(EvolutionAPIService $evolutionApiService, UploadUtil $util)
    {
        $this->evolutionApiService = $evolutionApiService;
        $this->util = $util;
        $this->middleware('permission:notificacao_cobranca_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:notificacao_cobranca_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:notificacao_cobranca_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:notificacao_cobranca_delete', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        $empresa = auth()->user()->empresa;
        $instance = 'USER_SLYM_' . auth()->user()->id;
        if ($empresa) {
            $instance = 'EMPRESA_SLYM_' . $empresa->id;
        }

        $status = $request->get('status');
        $gateway = $request->get('gateway');
        $empresa_id = $request->get('empresa');
        $prioridade = $request->get('prioridade');
        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $recorrente = $request->get('recorrente');

        $empresa = null;
        if ($empresa_id) {
            $empresa = Empresa::findOrFail($empresa_id);
        }

        $data = NotificacaoCobranca::orderBy('id', 'desc')
            ->when(!empty($start_date), function ($query) use ($start_date) {
                return $query->whereDate('data_vencimento', '>=', $start_date);
            })
            ->when(!empty($end_date), function ($query) use ($end_date,) {
                return $query->whereDate('data_vencimento', '<=', $end_date);
            })
            ->when($status != '', function ($query) use ($status) {
                return $query->where('status', $status);
            })
            ->when($gateway != '', function ($query) use ($gateway) {
                return $query->where('gateway', $gateway);
            })
            ->when($empresa != null, function ($query) use ($empresa) {
                return $query->where('empresa_id', $empresa->id);
            })
            ->when($recorrente != null, function ($query) use ($recorrente) {
                return $query->where('recorrente', $recorrente);
            })
            ->paginate(60);

        // dd(auth()->user());
        $responseWpp = @$this->evolutionApiService->instanceCheck($instance);
        $instanceNotExists = @$responseWpp['status'] == 404 && @$responseWpp['message'] == 'Instance not found';
        if ($instanceNotExists) {
            $this->evolutionApiService->createInstance($instance);
            $responseWpp = @$this->evolutionApiService->instanceCheck($instance);
        }

        $isConnected = @$responseWpp['instance']['state'] == 'open';
        $wppData = [];

        if ($isConnected) {
            $responseData = $this->evolutionApiService->instances($instance);
            $wppData['whatsapp'] = preg_replace('/[^0-9]/', '', $responseData[0]['ownerJid']);
            $wppData['name'] = $responseData[0]['profileName'];
        }

        return view('notificacao_cobrancas.index', compact('data', 'empresa', 'isConnected', 'wppData'));
    }

    public function create()
    {
        return view('notificacao_cobrancas.create');
    }

    public function edit($id)
    {
        $item = NotificacaoCobranca::findOrFail($id);
        return view('notificacao_cobrancas.edit', compact('item'));
    }

    public function show($id)
    {
        $item = NotificacaoCobranca::findOrFail($id);
        $item->visualizada = 1;
        $item->save();

        return view('notificacao_cobrancas.show', compact('item'));
    }

    public function store(Request $request)
    {
        $this->__validate($request);

        try {
            $empresa = auth()->user()->empresa;
            $request->merge([
                'empresa_id' => $empresa->id,
                'criador_id' => auth()->user()->id,
                'valor' => str_replace(',', '.', str_replace('.', '', $request->valor)),
            ]);
            NotificacaoCobranca::create($request->all());
            session()->flash("flash_success", "Notificação criada com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: ' . $e->getMessage());
        }
        return redirect()->route('notificacao-cobranca.index');
    }

    public function update(Request $request, $id)
    {
        $this->__validate($request);

        $item = NotificacaoCobranca::findOrFail($id);
        try {
            $empresa = auth()->user()->empresa;
            $request->merge([
                'empresa_id' => $empresa->id,
                'valor' => str_replace(',', '.', str_replace('.', '', $request->valor)),
            ]);

            $item->fill($request->all())->save();
            session()->flash("flash_success", "Notificação alterada com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: ' . $e->getMessage());
        }
        return redirect()->route('notificacao-cobranca.index');
    }

    public function destroy($id)
    {
        $item = NotificacaoCobranca::findOrFail($id);
        try {
            $item->delete();
            session()->flash("flash_success", "Notificação removida com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: ' . $e->getMessage());
        }
        return redirect()->route('notificacao-cobranca.index');
    }

    private function __validate(Request $request)
    {
        $rules = [
            'whatsapp' => 'required',
            'titulo' => 'required|max:30',
            'gateway' => 'required',
            'dias_antes_vencimento' => 'required',
            'dias_antes_vencimento' => 'required',
            'data_vencimento' => 'required',
        ];

        $messages = [
            'whatsapp.required' => 'Campo Obrigatório',
            'gateway.required' => 'Campo Obrigatório',
            'dias_antes_vencimento.required' => 'Campo Obrigatório',
            'data_vencimento.required' => 'Campo Obrigatório',

            'titulo.required' => 'Campo Obrigatório',
            'titulo.max' => 'Máximo de 30 caracteres',
        ];
        $this->validate($request, $rules, $messages);
    }

    public function destroySelecet(Request $request)
    {
        $removidos = 0;
        for ($i = 0; $i < sizeof($request->item_delete); $i++) {
            $item = NotificacaoCobranca::findOrFail($request->item_delete[$i]);
            try {
                $item->delete();
                $removidos++;
            } catch (\Exception $e) {
                session()->flash("flash_error", 'Algo deu errado: ' . $e->getMessage());
                return redirect()->route('notificacao-cobranca.index');
            }
        }

        session()->flash("flash_success", "Total de itens removidos: $removidos!");
        return redirect()->route('notificacao-cobranca.index');
    }

    public function conectar()
    {
        return view('notificacao_cobrancas.connect');
    }

    public function conectarAcao()
    {
        $request = new Request();

        try {
            $empresa = auth()->user()->empresa;
            $instance = 'USER_' . auth()->user()->id;
            if ($empresa) {
                $instance = 'EMPRESA_SLYM_' . $empresa->id;
            }

            $decodedData = $this->evolutionApiService->instanceCheck($instance);
            if (@$decodedData['status'] == 404) {
                $this->evolutionApiService->createInstance($instance, '');
                $decodedData = $this->evolutionApiService->instanceCheck($instance);
            }

            if (@$decodedData['instance']['state'] != 'open') {
                if ($request->start == 1) {
                    $this->evolutionApiService->instanceLogout($instance);
                    sleep(2);
                }

                $dataQrCode = $this->evolutionApiService->instanceConnect($instance);

                return response([
                    'status' => true,
                    'qr' => $dataQrCode,
                    'msg' => 'QRCode gerado com sucesso'
                ]);
            } else if (@$decodedData['instance']['state'] == 'open' || @$decodedData['status'] == 'WORKING') {
                return response([
                    'status' => 'connected',
                    'qr' => '',
                    'msg' => 'Seu dispositivo já está conectado!'
                ]);
            }
        } catch (\Throwable $th) {
            return response([
                'status' => false,
                'qr' => '',
                'msg' => 'Ocorreu uma falha entre em contato para solucionar!'
            ]);
        }

        return response([
            'status' => false,
            'qr' => '',
            'msg' => 'Ocorreu uma falha entre em contato para solucionar!'
        ]);
    }

    public function config()
    {
        $config = NotificacaoCobrancaConfig::where('empresa_id', auth()->user()->empresa->id)->first();
        return view('notificacao_cobrancas.config', compact('config'));
    }

    public function configSave(Request $request)
    {
        $request->validate([
            'frase_cobranca' => 'required|string|max:500',
            'public_key_cashin_paymundi' => 'nullable|string|max:250',
            'public_key_cashout_paymundi' => 'nullable|string|max:250',
            'secret_key_cashin_paymundi' => 'nullable|string|max:300',
            'secret_key_cashout_paymundi' => 'nullable|string|max:300',
            'client_id_onz' => 'nullable|string|max:250',
            'client_secret_onz' => 'nullable|string|max:250',
            'crt_onz' => 'nullable|file',
            'pfx_onz' => 'nullable|file',
            'key_onz' => 'nullable|file',
        ], [
            'frase_cobranca.required' => 'Campo Obrigatório',
            'frase_cobranca.string' => 'Campo deve ser um texto',
            'frase_cobranca.max' => 'Máximo de 500 caracteres',
            'public_key_cashin_paymundi.max' => 'Máximo de 250 caracteres',
            'public_key_cashout_paymundi.max' => 'Máximo de 250 caracteres',
            'secret_key_cashin_paymundi.max' => 'Máximo de 300 caracteres',
            'secret_key_cashout_paymundi.max' => 'Máximo de 300 caracteres',
            'client_id_onz.max' => 'Máximo de 250 caracteres',
            'client_secret_onz.max' => 'Máximo de 250 caracteres',
            'crt_onz.mimes' => 'O arquivo deve ser um certificado .crt',
            'pfx_onz.mimes' => 'O arquivo deve ser um certificado .pfx',
            'key_onz.mimes' => 'O arquivo deve ser um certificado .key',
        ]);

        $config = NotificacaoCobrancaConfig::where('empresa_id', auth()->user()->empresa->id)->first();
        if (!$config) {
            $config = new NotificacaoCobrancaConfig();
            $config->empresa_id = auth()->user()->empresa->id;
        }
        $config->public_key_cashin_paymundi = $request->input('public_key_cashin_paymundi');
        $config->public_key_cashout_paymundi = $request->input('public_key_cashout_paymundi');
        $config->secret_key_cashin_paymundi = $request->input('secret_key_cashin_paymundi');
        $config->secret_key_cashout_paymundi = $request->input('secret_key_cashout_paymundi');
        $config->client_id_onz = $request->input('client_id_onz');
        $config->client_secret_onz = $request->input('client_secret_onz');

        if ($request->hasFile('crt_onz')) {
            $file_name = $this->util->uploadFile($request->crt_onz, '/certificados');
            $crtPath = 'uploads/certificados/' . $file_name;
            $config->crt_onz = $crtPath;
        }

        if ($request->hasFile('pfx_onz')) {
            $file_name = $this->util->uploadFile($request->pfx_onz, '/certificados');
            $pfxPath = 'uploads/certificados/' . $file_name;
            $config->pfx_onz = $pfxPath;
        }

        if ($request->hasFile('key_onz')) {
            $file_name = $this->util->uploadFile($request->key_onz, '/certificados');
            $keyPath = 'uploads/certificados/' . $file_name;
            $config->key_onz = $keyPath;
        }

        $config->senha_onz = $request->input('senha_onz');
        $config->frase_cobranca = $request->input('frase_cobranca');
        $config->chave_pix = $request->input('chave_pix');
        $config->save();
        session()->flash("flash_success", "Configurações salvas com sucesso!");
        return redirect()->route('notificacao-cobranca.config');
    }
}
