<?php

namespace App\Filament\User\Resources;

use App\Filament\User\Resources\ClientResource\Pages;
use App\Filament\User\Resources\ClientResource\RelationManagers;
use App\Models\Client;
use Filament\Forms;
use Filament\Forms\Components\TextInput;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\BulkActionGroup;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Actions\DeleteAction;
use Filament\Notifications\Notification;


class ClientResource extends Resource
{
    public static function getNavigationLabel(): string
    {
        return 'Clientes';
    }

    public static function getModelLabel(): string
    {
        return 'Cliente';
    }

    public static function getPluralModelLabel(): string
    {
        return 'Clientes';
    }

    protected static ?string $modelLabel = 'Cliente';

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        $data['tenant_id'] = auth()->user()->tenant_id;
        return $data;
    }
    
    protected static ?string $model = Client::class;
    
    protected static ?string $navigationIcon = 'heroicon-o-user-group';
    protected static ?string $navigationGroup = 'Cadastros';
    protected static ?int $navigationSort = 1;
    protected static ?string $pluralLabel = 'Clientes';

    // public static function shouldRegisterNavigation(): bool
    // {
    //     return auth()->user()->canAccessModule('clients');
    // }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('name')
                    ->label('Cliente Nome')
                    ->maxLength(50)
                    ->required(),
                TextInput::make('phone')
                    ->label('Telefone')
                    ->mask('(99) 99999-9999')
                    ->tel()
                    ->required(),
                TextInput::make('email')
                    ->label('Cliente Email')
                    ->maxLength(50)
                    ->required()
                    ->email()
                    ->unique(ignoreRecord: true),          
                // PEGA APENAS O TENANT DO USUARIO LOGADO
                Select::make('tenant_id')
                    ->label('Empresa do Cliente')
                    ->relationship('tenant', 'name', fn ($query) => $query->where('id', auth()->user()->tenant_id))
                    ->required()
                    ->preload()
                    ->searchable()
                    ->default(auth()->user()->tenant_id)
                    ->disabled()
                    ->dehydrated(), // NAO PODE SER ALTERADO
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            // pega apenas os clientes do tenant do usuario logado
            ->modifyQueryUsing(fn (Builder $query) => $query->where('tenant_id', auth()->user()->tenant_id))
            ->columns([
                TextColumn::make('name')
                    ->label('Cliente Nome')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                TextColumn::make('email')
                    ->label('Cliente Email')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                TextColumn::make('phone')
                    ->label('Cliente Telefone')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                TextColumn::make('tenant.name')
                    ->label('Empresa')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: false),
                TextColumn::make('created_at')
                    ->label('Data de Criação')
                    ->dateTime('d/m/Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('updated_at')
                    ->label('Data de Atualização')
                    ->dateTime('d/m/Y')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                //
            ])
            ->actions([
                EditAction::make()
                    ->tooltip('Editar')
                    ->iconButton()
                    ->color('warning')
                    ->icon('heroicon-o-pencil-square'),
                DeleteAction::make()
                    ->tooltip('Excluir')
                    ->iconButton()
                    ->color('danger')
                    ->icon('heroicon-o-trash')
                    ->modalHeading('Excluir Cliente')
                    ->modalDescription('Tem certeza que deseja excluir este cliente? Esta ação não poderá ser desfeita.')
                    ->modalSubmitActionLabel('Sim, excluir')
                    ->modalCancelActionLabel('Não, cancelar')
                    ->successNotification(null)
                    ->after(function() {
                        Notification::make()
                            ->success()
                            ->color('success')
                            ->icon('heroicon-o-check-circle')
                            ->title('Cliente excluído!')
                            ->body('O cliente foi removido com sucesso')
                            ->send();
                    }),
            ])
            ->bulkActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->label('Excluir Selecionados')
                        ->modalHeading('Excluir Clientes Selecionados')
                        ->modalDescription('Tem certeza que deseja excluir os clientes selecionados? Esta ação não poderá ser desfeita.')
                        ->modalSubmitActionLabel('Sim, excluir')
                        ->modalCancelActionLabel('Não, cancelar')
                        ->successNotification(null)
                        ->after(function() {
                            Notification::make()
                                ->success()
                                ->color('success')
                                ->icon('heroicon-o-check-circle')
                                ->title('Clientes excluídos!')
                                ->body('Os clientes foram removidos com sucesso')
                                ->send();
                        })
                ])
                ->label('Ações em massa'),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListClients::route('/'),
            'create' => Pages\CreateClient::route('/create'),
            'edit' => Pages\EditClient::route('/{record}/edit'),
        ];
    }
}
