<?php
session_start();
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Unauthorized']);
    exit;
}

require_once '../config/database.php';

try {
    $data = [];
    
    // Recebimentos por mês (próximos 12 meses)
    $stmt = $pdo->prepare("
        SELECT 
            DATE_FORMAT(i.due_date, '%Y-%m') as month,
            SUM(CASE WHEN i.status = 'pending' THEN i.amount ELSE 0 END) as pending_amount,
            SUM(CASE WHEN i.status = 'paid' THEN i.amount ELSE 0 END) as paid_amount,
            SUM(CASE WHEN i.status = 'overdue' THEN i.amount ELSE 0 END) as overdue_amount
        FROM installments i
        JOIN movements m ON i.movement_id = m.id
        WHERE i.due_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 12 MONTH)
        AND m.user_id = ?
        GROUP BY DATE_FORMAT(i.due_date, '%Y-%m')
        ORDER BY month ASC
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $monthly = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Garantir que todos os próximos 12 meses estejam presentes
    $data['monthly'] = [];
    $current = new DateTime();
    for ($i = 0; $i < 12; $i++) {
        $monthKey = $current->format('Y-m');
        $monthData = array_filter($monthly, function($item) use ($monthKey) {
            return $item['month'] === $monthKey;
        });
        
        $data['monthly'][] = [
            'month' => $monthKey,
            'pending_amount' => empty($monthData) ? 0 : current($monthData)['pending_amount'],
            'paid_amount' => empty($monthData) ? 0 : current($monthData)['paid_amount'],
            'overdue_amount' => empty($monthData) ? 0 : current($monthData)['overdue_amount']
        ];
        
        $current->modify('+1 month');
    }

    // Status geral das parcelas
    $stmt = $pdo->prepare("
        SELECT 
            i.status,
            COUNT(*) as total_count,
            SUM(i.amount) as total_amount
        FROM installments i
        JOIN movements m ON i.movement_id = m.id
        WHERE m.user_id = ?
        GROUP BY i.status
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $statusData = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Garantir que todos os status estejam presentes
    $data['status_summary'] = [];
    $allStatus = ['pending', 'paid', 'overdue'];
    foreach ($allStatus as $status) {
        $statusInfo = array_filter($statusData, function($item) use ($status) {
            return $item['status'] === $status;
        });
        
        $data['status_summary'][] = [
            'status' => $status,
            'total_count' => empty($statusInfo) ? 0 : current($statusInfo)['total_count'],
            'total_amount' => empty($statusInfo) ? 0 : current($statusInfo)['total_amount']
        ];
    }

    // Top 5 clientes com mais recebimentos pendentes
    $stmt = $pdo->prepare("
        SELECT 
            c.name as client_name,
            SUM(CASE WHEN i.status IN ('pending', 'overdue') THEN i.amount ELSE 0 END) as pending_amount
        FROM clients c
        JOIN movements m ON m.client_id = c.id
        JOIN installments i ON i.movement_id = m.id
        WHERE i.status IN ('pending', 'overdue')
        AND m.user_id = ?
        GROUP BY c.id, c.name
        HAVING pending_amount > 0
        ORDER BY pending_amount DESC
        LIMIT 5
    ");
    $stmt->execute([$_SESSION['user_id']]);
    $data['top_clients'] = $stmt->fetchAll(PDO::FETCH_ASSOC);

    header('Content-Type: application/json');
    echo json_encode($data);
} catch (PDOException $e) {
    header('Content-Type: application/json');
    echo json_encode(['error' => $e->getMessage()]);
}
