<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\CategoriaProduto;
use App\Models\Empresa;
use App\Models\FaturaPreVenda;
use App\Models\Funcionario;
use App\Models\ItemPreVenda;
use App\Models\NaturezaOperacao;
use App\Models\PreVenda;
use App\Models\Produto;
use App\Models\Nfce;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use StringBackedEnum;
use Svg\Tag\Rect;
use Illuminate\Support\Str;
use NFePHP\DA\NFe\CupomNaoFiscal;

class PreVendaController extends Controller
{

    public function __construct()
    {
        $this->middleware('permission:pre_venda_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:pre_venda_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:pre_venda_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:pre_venda_delete', ['only' => ['destroy']]);
    }

    public function index(Request $request)
    {
        $locais = __getLocaisAtivoUsuario();
        $locais = $locais->pluck(['id']);

        $start_date = $request->get('start_date');
        $end_date = $request->get('end_date');
        $cliente_id = $request->get('cliente_id');
        $status = $request->get('status');
        $local_id = $request->get('local_id');

        $data = PreVenda::where('empresa_id', request()->empresa_id)
        ->when(!empty($cliente_id), function ($query) use ($cliente_id) {
            return $query->where('cliente_id', $cliente_id);
        })
        ->when(!empty($start_date), function ($query) use ($start_date) {
            return $query->whereDate('created_at', '>=', $start_date);
        })
        ->when(!empty($end_date), function ($query) use ($end_date,) {
            return $query->whereDate('created_at', '<=', $end_date);
        })
        ->when(!empty($status), function ($query) use ($status) {
            if ($status == -1) {
                return $query->where('status', '!=', 1);
            } else {
                return $query->where('status', $status);
            }
        })
        ->when($local_id, function ($query) use ($local_id) {
            return $query->where('local_id', $local_id);
        })
        ->when(!$local_id, function ($query) use ($locais) {
            return $query->whereIn('local_id', $locais);
        })
        ->orderBy('id', 'desc')
        ->paginate(env("PAGINACAO"));
        return view('pre_venda.index', compact('data'));
    }


    public function create()
    {
        if (!__isCaixaAberto()) {
            session()->flash("flash_warning", "Abrir caixa antes de continuar!");
            return redirect()->route('caixa.create');
        }

        $abertura = Caixa::where('usuario_id', get_id_user())
        ->where('status', 1)
        ->first();

        $categorias = CategoriaProduto::where('empresa_id', request()->empresa_id)->get();
        $funcionarios = Funcionario::where('empresa_id', request()->empresa_id)->get();
        $naturezas = NaturezaOperacao::where('empresa_id', request()->empresa_id)->get();
        if (sizeof($naturezas) == 0) {
            session()->flash("flash_warning", "Primeiro cadastre um natureza de operação!");
            return redirect()->route('natureza-operacao.create');
        }
        $caixa = __isCaixaAberto();

        $tiposPagamento = Nfce::tiposPagamento();
        // dd($tiposPagamento);
        $config = Empresa::findOrFail(request()->empresa_id);

        $config = Empresa::findOrFail(request()->empresa_id);
        if($config == null){
            session()->flash("flash_warning", "Configure antes de continuar!");
            return redirect()->route('config.index');
        }

        if($config->natureza_id_pdv == null){
            session()->flash("flash_warning", "Configure a natureza de operação padrão para continuar!");
            return redirect()->route('config.index');
        }
        
        if($config != null && $config->tipos_pagamento_pdv){
            $tipos_pagamento_pdv = json_decode($config->tipos_pagamento_pdv, true);
            if(is_array($tipos_pagamento_pdv) && sizeof($tipos_pagamento_pdv) > 0){
                $temp = [];
                foreach($tiposPagamento as $key => $t){
                    if(in_array($t, $tipos_pagamento_pdv)){
                        $temp[$key] = $t;
                    }
                }
                $tiposPagamento = $temp;
            }
        }

        return view('pre_venda.create', compact('abertura', 'categorias', 'funcionarios', 'naturezas', 'caixa', 'tiposPagamento'));
    }

    public function store(Request $request)
    {
        if(!$request->produto_id){
            session()->flash("flash_error", "Inclua ao menos 1 item na pré venda");
            
            // Se for uma requisição AJAX, retornar erro em JSON (igual ao PDV)
            if ($request->ajax()) {
                return response()->json("Inclua ao menos 1 item na pré venda", 401);
            }
            
            return redirect()->back();
        }
        \Log::info('funcionario_id recebido:', [$request->funcionario_id]);
        try {
            // $valor_total = $this->somaItens($request);

            $natureza = NaturezaOperacao::where('empresa_id', request()->empresa_id)->first();
            $caixa = __isCaixaAberto();
            $request->merge([
                'cliente_id' => $request->cliente_id,
                'bandeira_cartao' => $request->bandeira_cartao ?? '',
                'cnpj_cartao' => $request->cnpj_cartao ?? '',
                'cAut_cartao' => $request->cAut_cartao ?? '',
                'descricao_pag_outros' => $request->descricao_pag_outros ?? '',
                'rascunho' => $request->rascunho ?? 0,
                'usuario_id' => get_id_user(),
                'observacao' => $request->observacao ?? '',
                'qtd_volumes' => $request->qtd_volumes ?? 0,
                'peso_liquido' => $request->peso_liquido ?? 0,
                'funcionario_id' => $request->funcionario_id ?? null,
                'peso_bruto' => $request->peso_bruto ?? 0,
                'desconto' => $request->desconto ? __convert_value_bd($request->desconto) : 0,
                'valor_total' => __convert_value_bd($request->valor_total),
                'acrescimo' => $request->acrescimo ? __convert_value_bd($request->acrescimo) : 0,
                'natureza_id' => $natureza->id,
                'forma_pagamento' => '',
                'tipo_pagamento' => $request->tipo_pagamento_row ? '99' : $request->tipo_pagamento,
                'nome' => $request->nome,
                'cpf' => $request->cpf ?? '',
                'local_id' => $caixa->local_id,
                'codigo' => $this->gerarCodigoPreVenda()
            ]);
            // dd($request->all());
            
            $preVenda = PreVenda::create($request->all());
            
            for ($i = 0; $i < sizeof($request->produto_id); $i++) {
                $product = Produto::find($request->produto_id[$i]);
                
                // Verificar se o produto existe
                if (!$product) {
                    throw new \Exception("Produto com ID {$request->produto_id[$i]} não encontrado.");
                }
                
                $cfop = 0;
                ItemPreVenda::create([
                    'pre_venda_id' => $preVenda->id,
                    'produto_id' => (int)$request->produto_id[$i],
                    'quantidade' => __convert_value_bd($request->quantidade[$i]),
                    'valor' => __convert_value_bd($request->valor_unitario[$i]),
                    'cfop' => $cfop,
                    'observacao' => $request->observacao ?? '',
                ]);
            }

            if ($request->tipo_pagamento_row) {
                for ($i = 0; $i < sizeof($request->tipo_pagamento_row); $i++) {
                    FaturaPreVenda::create([
                        'valor_parcela' => __convert_value_bd($request->valor_integral_row[$i]),
                        'tipo_pagamento' => $request->tipo_pagamento_row[$i],
                        'pre_venda_id' => $preVenda->id,
                        'vencimento' => $request->data_vencimento_row[$i] ?? date('Y-m-d')
                    ]);
                }
            } else {
                FaturaPreVenda::create([
                    'valor_parcela' => __convert_value_bd($request->valor_total),
                    'tipo_pagamento' => $request->tipo_pagamento,
                    'pre_venda_id' => $preVenda->id,
                    'vencimento' => $request->data_vencimento ?? date('Y-m-d')
                ]);
            }
            session()->flash("flash_success", "Pré venda realizada com sucesso!");
        } catch (\Exception $e) {
            // echo $e->getMessage() . '<br>' . $e->getLine();
            // die;
            session()->flash("flash_error", "Algo deu errado por aqui: " . $e->getMessage());
            
            // Se for uma requisição AJAX, retornar erro em JSON (igual ao PDV)
            if ($request->ajax()) {
                return response()->json($e->getMessage() . ", line: " . $e->getLine() . ", file: " . $e->getFile(), 401);
            }
        }
        
        // Se for uma requisição AJAX, retornar o objeto da pré-venda (igual ao PDV)
        if ($request->ajax()) {
            return response()->json($preVenda);
        }
        
        return redirect()->back()->with(['codigo' => $preVenda->codigo]);
    }

    public function imprimir($codigo){
        $item = PreVenda::with(['itens.produto', 'fatura', 'cliente'])->where('codigo', $codigo)
        ->where('empresa_id', request()->empresa_id)
        ->first();
        
        if (!$item) {
            abort(404, 'Pré-venda não encontrada');
        }
        
        $config = Empresa::where('id', $item->empresa_id)
        ->first();
        
        // Preparar dados para o cupom (igual ao PDV)
        $item->itens = $item->itens->map(function($itemVenda) {
            // Calcular subtotal (campo não existe na tabela, então calculamos)
            $itemVenda->sub_total = $itemVenda->quantidade * $itemVenda->valor;
            
            // Garantir que valor_unitario existe
            $itemVenda->valor_unitario = $itemVenda->valor;
            
            return $itemVenda;
        });
        
        // Preparar faturas para o cupom
        if ($item->fatura->count() > 0) {
            $item->fatura = $item->fatura->map(function($fatura) {
                // Garantir que o valor está correto
                $fatura->valor = $fatura->valor_parcela;
                return $fatura;
            })->filter(function($fatura) {
                return $fatura->valor_parcela > 0;
            });
        }
        
        $cupom = new CupomNaoFiscal($item, $config, 1);

        $pdf = $cupom->render();
        return response($pdf)
        ->header('Content-Type', 'application/pdf');
    }

    private function gerarCodigoPreVenda()
    {
        // Buscar a última pré-venda da empresa
        $ultimaPreVenda = PreVenda::where('empresa_id', request()->empresa_id)
            ->where('codigo', 'like', 'PRV%')
            ->orderBy('id', 'desc')
            ->first();
        
        if ($ultimaPreVenda) {
            // Extrair o número do último código
            $numero = (int) substr($ultimaPreVenda->codigo, 3);
            $numero++;
        } else {
            // Se não existir nenhuma, começar com 1
            $numero = 1;
        }
        
        // Formatar com zeros à esquerda (4 dígitos)
        return 'PRV' . str_pad($numero, 4, '0', STR_PAD_LEFT);
    }

    private function somaItens($request)
    {
        $valor_total = 0;
        for ($i = 0; $i < sizeof($request->produto_id); $i++) {
            $valor_total += __convert_value_bd($request->subtotal_item[$i]);
        }
        return $valor_total;
    }

    public function destroy($id)
    {
        $item = PreVenda::with(['itens.produto'])->findOrFail($id);
        
        // Verificar se a pré-venda pode ser excluída
        if ($item->status == 0) {
            // Pré-venda já foi transformada em venda, não pode ser excluída
            session()->flash("flash_error", "Esta pré-venda não pode ser excluída pois já foi transformada em venda (Status: Finalizada).");
            return redirect()->route('pre-venda.index');
        }
        
        try {
            // Restaurar estoque antes de deletar
            foreach ($item->itens as $itemVenda) {
                if ($itemVenda->produto && $itemVenda->produto->gerenciar_estoque) {
                    // Usar o EstoqueUtil para restaurar o estoque
                    $estoqueUtil = app(\App\Utils\EstoqueUtil::class);
                    $estoqueUtil->incrementaEstoque(
                        $itemVenda->produto_id,
                        $itemVenda->quantidade,
                        $itemVenda->variacao_id ?? null,
                        $item->local_id
                    );
                    
                    // Registrar movimentação de estoque (histórico)
                    $estoqueUtil->movimentacaoProduto(
                        $itemVenda->produto_id,
                        $itemVenda->quantidade,
                        'incremento',
                        $item->id,
                        'exclusao_pre_venda',
                        get_id_user(),
                        $itemVenda->variacao_id ?? null
                    );
                }
            }
            
            $item->delete();
            session()->flash("flash_success", "Pré-venda removida com sucesso! Estoque restaurado.");
        } catch (\Exception $e) {
            session()->flash("flash_error", "Algo deu errado: " . $e->getMessage());
        }
        return redirect()->route('pre-venda.index');
    }

    /**
     * Marca uma pré-venda como finalizada (transformada em venda)
     */
    public function marcarComoFinalizada($id, $venda_id = null)
    {
        try {
            $preVenda = PreVenda::findOrFail($id);
            
            $preVenda->update([
                'status' => 0, // 0 = finalizada
                'venda_id' => $venda_id // ID da venda gerada
            ]);
            
            return response()->json(['success' => true, 'message' => 'Pré-venda marcada como finalizada']);
            
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Erro ao marcar pré-venda como finalizada: ' . $e->getMessage()], 500);
        }
    }
}
