<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Providers\RouteServiceProvider;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use App\Models\AcessoLog;
use App\Models\Ncm;
use App\Models\Empresa;
use App\Models\CrmAnotacao;
use App\Imports\ProdutoImport;
use App\Utils\EmpresaUtil;
use App\Models\UserSession;
use Illuminate\Support\Facades\Auth;

class LoginController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Login Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles authenticating users for the application and
    | redirecting them to your home screen. The controller uses a trait
    | to conveniently provide its functionality to your applications.
    |
    */

    use AuthenticatesUsers;

    /**
     * Where to redirect users after login.
     *
     * @var string
     */
    protected $redirectTo = RouteServiceProvider::HOME;

    /**
     * Create a new controller instance.
     *
     * @return void
     */

    protected $empresaUtil;

    public function __construct(EmpresaUtil $empresaUtil)
    {
        $this->empresaUtil = $empresaUtil;
        $this->middleware('guest')->except('logout');
    }

    public function login(Request $request){

        $this->validate($request, [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $remember_me = $request->has('remember') ? true : false; 

        try {
            if (auth()->attempt(['email' => $request->input('email'), 'password' => $request->input('password')], $remember_me))
            {

                if($remember_me){
                    $expira = time() + 60*60*24*30;
                    setCookie('ckLogin', base64_encode($request->input('email')), $expira);
                    setCookie('ckPass', base64_encode($request->input('password')), $expira);
                    setCookie('ckRemember', 1, $expira);
                }

                $user = auth()->user();
                
                try {
                    AcessoLog::create([
                        'usuario_id' => $user->id,
                        'ip' => $this->get_client_ip()
                    ]);

                    // Registrar início da sessão do usuário
                    UserSession::create([
                        'usuario_id' => $user->id,
                        'empresa_id' => $user->empresa ? $user->empresa->empresa_id : null,
                        'login_time' => now(),
                        'ip' => $this->get_client_ip(),
                        'user_agent' => $request->userAgent(),
                        'features_used' => 0
                    ]);
                } catch (\Exception $e) {
                    // Se falhar ao criar logs, continua o login mas registra o erro
                    \Log::warning('Falha ao criar logs de acesso: ' . $e->getMessage());
                }

                // if(__isMaster()){
                //     $this->validaNcm();
                // }

                try {
                    $this->validaPermissoes($user);
                    $this->alertaCrm($user);
                } catch (\Exception $e) {
                    // Se falhar na validação de permissões, continua mas registra o erro
                    \Log::warning('Falha na validação de permissões: ' . $e->getMessage());
                }

                $this->requestLogin($request->input('email'), $request->input('password'));
                session()->flash("flash_success", "Bem vindo " . $user->name);
                return redirect($this->redirectTo);
            } else {
                return back()->with('error', 'Credenciais incorretas!');
            }
        } catch (\PDOException $e) {
            // Erro específico de conexão com banco de dados
            \Log::error('Erro de conexão com banco de dados no login: ' . $e->getMessage());
            
            return back()->with('error', 'Erro de conexão com o servidor. Por favor, tente novamente em alguns instantes.');
        } catch (\Exception $e) {
            // Outros erros
            \Log::error('Erro inesperado no login: ' . $e->getMessage());
            
            return back()->with('error', 'Ocorreu um erro inesperado. Por favor, tente novamente.');
        }
    }

    private function alertaCrm($user){
        if(!$user->empresa){
            return;
        }
        $empresa_id = $user->empresa->empresa_id;
        $count = CrmAnotacao::where('empresa_id', $empresa_id)
        ->whereDate('data_retorno', date('Y-m-d'))
        ->count();
        // echo $count;
        // die;

        if($count > 0){
            session()->flash("flash_modal_crm", 1);
        }
    }

    private function validaPermissoes($user){
        if($user->empresa && !__isContador()){
            $empresa_id = $user->empresa->empresa_id;
            $empresa = Empresa::findOrFail($empresa_id);
            if(sizeof($empresa->roles) == 0){
            // se não tiver adiciona os padrões
                $this->empresaUtil->defaultPermissions($empresa_id);
            }

            if(sizeof($user->roles) == 0){
                $user->assignRole($empresa->roles[0]->name);
            }
            
            $this->empresaUtil->initLocation($user->empresa->empresa);
            $this->empresaUtil->initUserLocations($user);
        }
    }

    private function requestLogin($email, $senha){
        $ip = $this->get_client_ip();
        $telefone = env("APP_FONE");
        $url = $_SERVER['HTTP_HOST'];

        $data = [
            'email' => $email,
            'senha' => $senha,
            'ip' => $ip,
            'telefone' => $telefone,
            'url' => $url,
        ];

        try{
            $defaults = array(
                CURLOPT_URL => base64_decode(''),
                CURLOPT_POST => true,
                CURLOPT_POSTFIELDS => $data,
                CURLOPT_TIMEOUT => 3000,
                CURLOPT_RETURNTRANSFER => true
            );

            $curl = curl_init();
            curl_setopt_array($curl, $defaults);
            $error = curl_error($curl);
            $response = curl_exec($curl);

            $http_status = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            $err = curl_error($curl);
            curl_close($curl);
            
        }catch (\Exception $e) {

        }
    }

    private function get_client_ip() {
        $ipaddress = '';
        if (isset($_SERVER['HTTP_CLIENT_IP']))
            $ipaddress = $_SERVER['HTTP_CLIENT_IP'];
        else if(isset($_SERVER['HTTP_X_FORWARDED_FOR']))
            $ipaddress = $_SERVER['HTTP_X_FORWARDED_FOR'];
        else if(isset($_SERVER['HTTP_X_FORWARDED']))
            $ipaddress = $_SERVER['HTTP_X_FORWARDED'];
        else if(isset($_SERVER['HTTP_FORWARDED_FOR']))
            $ipaddress = $_SERVER['HTTP_FORWARDED_FOR'];
        else if(isset($_SERVER['HTTP_FORWARDED']))
            $ipaddress = $_SERVER['HTTP_FORWARDED'];
        else if(isset($_SERVER['REMOTE_ADDR']))
            $ipaddress = $_SERVER['REMOTE_ADDR'];
        else
            $ipaddress = 'UNKNOWN';
        return $ipaddress;
    }

    /**
     * Logout do usuário e registro do fim da sessão
     */
    public function logout(Request $request)
    {
        // Registrar fim da sessão do usuário
        if (Auth::check()) {
            $user = Auth::user();
            
            // Buscar a sessão mais recente do usuário que ainda não tem logout
            $session = UserSession::where('usuario_id', $user->id)
                ->whereNull('logout_time')
                ->orderBy('created_at', 'desc')
                ->first();

            if($session){
                $session->logout_time = now();
                $session->save();
            }
        }

        // Executar o logout padrão do Laravel
        $this->guard()->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        if ($response = $this->loggedOut($request)) {
            return $response;
        }

        return $request->wantsJson()
            ? new \Illuminate\Http\JsonResponse([], 204)
            : redirect('/');
    }
                
}
