<?php

namespace App\Exceptions;

use Illuminate\Foundation\Exceptions\Handler as ExceptionHandler;
use Throwable;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\ErroCriticoNotificacao;
use App\Services\NotificacaoService;
use Symfony\Component\HttpKernel\Exception\HttpException;
use Illuminate\Auth\AuthenticationException;
use Illuminate\Session\TokenMismatchException;
use Illuminate\Validation\ValidationException;

class Handler extends ExceptionHandler
{
    /**
     * A list of the exception types that are not reported.
     *
     * @var array
     */
    protected $dontReport = [
        AuthenticationException::class,
        HttpException::class,
        TokenMismatchException::class,
        ValidationException::class,
    ];

    /**
     * A list of the inputs that are never flashed for validation exceptions.
     *
     * @var array
     */
    protected $dontFlash = [
        'password',
        'password_confirmation',
    ];

    /**
     * Register the exception handling callbacks for the application.
     *
     * @return void
     */
    public function register()
    {
        $this->reportable(function (Throwable $e) {
            $this->reportarErroCritico($e);
        });

        // Tratamento específico para erros de conexão com banco de dados
        $this->renderable(function (\PDOException $e, $request) {
            // Verifica se é um erro de conexão com banco de dados
            if (str_contains($e->getMessage(), 'Nenhuma conexão pôde ser feita') || 
                str_contains($e->getMessage(), 'Connection refused') ||
                str_contains($e->getMessage(), 'SQLSTATE[HY000] [2002]') ||
                str_contains($e->getMessage(), 'SQLSTATE[HY000] [2006]') ||
                str_contains($e->getMessage(), 'SQLSTATE[HY000] [2013]')) {
                
                \Log::critical('Erro de conexão com banco de dados detectado: ' . $e->getMessage(), [
                    'url' => $request->fullUrl(),
                    'method' => $request->method(),
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent()
                ]);

                // Se for uma requisição AJAX, retorna JSON
                if ($request->expectsJson()) {
                    return response()->json([
                        'error' => 'Erro de conexão com o servidor',
                        'message' => 'Não foi possível conectar ao banco de dados. Tente novamente em alguns instantes.',
                        'retry_after' => 30
                    ], 503);
                }

                // Para requisições normais, retorna a view de erro 500 personalizada
                return response()->view('errors.500', [], 503);
            }
        });

        // Tratamento para outros erros de banco de dados
        $this->renderable(function (\Illuminate\Database\QueryException $e, $request) {
            \Log::error('Erro de consulta no banco de dados: ' . $e->getMessage(), [
                'url' => $request->fullUrl(),
                'method' => $request->method(),
                'sql' => $e->getSql() ?? 'N/A'
            ]);

            // Se for uma requisição AJAX, retorna JSON
            if ($request->expectsJson()) {
                return response()->json([
                    'error' => 'Erro no banco de dados',
                    'message' => 'Ocorreu um erro ao processar sua solicitação. Tente novamente.'
                ], 500);
            }

            // Para requisições normais, retorna a view de erro 500 personalizada
            return response()->view('errors.500', [], 500);
        });
    }
    
    /**
     * Avalia e reporta erros críticos por email
     *
     * @param Throwable $e
     * @return void
     */
    protected function reportarErroCritico(Throwable $e)
    {
        // Lista de classes de exceção que são consideradas críticas
        $excecoesCriticas = [
            \PDOException::class, // Erros de banco de dados
            \ErrorException::class, // Erros graves do PHP
            \ParseError::class, // Erros de parse
            \TypeError::class, // Erros de tipo
        ];
        
        // Verifica se a exceção é crítica
        $isCritico = false;
        foreach ($excecoesCriticas as $tipo) {
            if ($e instanceof $tipo) {
                $isCritico = true;
                break;
            }
        }
        
        // Se não for crítica, retorna
        if (!$isCritico) {
            return;
        }
        
        // Log do erro crítico
        Log::critical('Erro crítico capturado:', [
            'message' => $e->getMessage(),
            'exception' => get_class($e),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString(),
        ]);
        
        try {
            // Obtém a lista de destinatários
            $destinatarios = NotificacaoService::getDestinatarios();
            
            // Prepara dados para a notificação
            $dados = [
                'mensagem' => $e->getMessage(),
                'classe' => get_class($e),
                'arquivo' => $e->getFile(),
                'linha' => $e->getLine(),
                'trace' => $e->getTraceAsString(),
                'data' => now()->format('d/m/Y H:i:s'),
                'url' => request()->fullUrl(),
                'user_agent' => request()->userAgent(),
                'ip' => request()->ip(),
            ];
            
            // Envia notificação para cada destinatário
            foreach ($destinatarios as $dest) {
                Mail::to($dest)->send(new ErroCriticoNotificacao($dados));
            }
        } catch (\Exception $emailError) {
            // Não podemos usar o sistema de email para reportar falhas no email
            Log::critical('Falha ao enviar notificação de erro crítico por email:', [
                'error' => $emailError->getMessage()
            ]);
        }
    }
}
