<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class NPSConfiguracao extends Model
{
    use HasFactory;
    
    protected $table = 'nps_configuracoes';
    
    protected $fillable = [
        'ativo',
        'data_inicio',
        'data_fim',
        'dia_mes',
        'titulo',
        'descricao'
    ];
    
    protected $casts = [
        'ativo' => 'boolean',
        'data_inicio' => 'date',
        'data_fim' => 'date',
        'dia_mes' => 'integer'
    ];
    
    /**
     * Verifica se a pesquisa NPS deve ser exibida hoje
     * 
     * @param int $usuarioId ID do usuário
     * @return bool
     */
    public static function deveExibirHoje($usuarioId)
    {
        $config = self::first();
        
        // Se não tiver configuração ou não estiver ativo, não exibe
        if (!$config || !$config->ativo) {
            return false;
        }
        
        $hoje = Carbon::now();
        
        // Verifica se já passou da data de início
        if ($config->data_inicio && $hoje->lt($config->data_inicio)) {
            return false;
        }
        
        // Verifica se já passou da data de término (se houver)
        if ($config->data_fim && $hoje->gt($config->data_fim)) {
            return false;
        }
        
        // Verifica se o usuário já respondeu ou ignorou no mês atual
        $primeiroDiaMes = Carbon::now()->startOfMonth();
        $mesAtual = $hoje->format('Y-m');
        
        $pesquisaUsuario = \App\Models\NPSPesquisa::where('usuario_id', $usuarioId)
            ->where('mes_referencia', $primeiroDiaMes->format('Y-m-d'))
            ->where(function($query) {
                $query->where('respondido', true)
                      ->orWhere('ignorado', true);
            })
            ->first();
            
        if (!$pesquisaUsuario) {
            // Se o usuário não respondeu nem ignorou no mês atual, exibe a pesquisa todos os dias
            return true;
        } else {
            // Se já respondeu ou ignorou, verificamos se:
            // 1. Estamos em um mês diferente da última resposta
            // 2. Já passamos do dia configurado para exibição
            
            $ultimaResposta = Carbon::parse($pesquisaUsuario->mes_referencia);
            $ultimoMesResposta = $ultimaResposta->format('Y-m');
            
            if ($mesAtual != $ultimoMesResposta && $hoje->day >= $config->dia_mes) {
                // Estamos em um novo mês e já passamos do dia configurado
                return true;
            }
        }
        
        // Se não se encaixar em nenhuma das condições acima, não exibe
        return false;
    }
    
    /**
     * Obter a configuração atual ou criar uma padrão se não existir
     * 
     * @return NPSConfiguracao
     */
    public static function obterOuCriar()
    {
        $config = self::first();
        
        if (!$config) {
            $config = self::create([
                'ativo' => false,
                'dia_mes' => 1,
                'titulo' => 'Pesquisa de Satisfação'
            ]);
        }
        
        return $config;
    }
}
