<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Consignacao extends Model
{
    use HasFactory;

    protected $table = 'consignacoes';

    protected $fillable = [
        'empresa_id',
        'usuario_id',
        'cliente_id',
        'funcionario_id',
        'numero_consignacao',
        'data_consignacao',
        'data_vencimento',
        'valor_total',
        'observacao',
        'status', // 'ativa', 'finalizada', 'cancelada'
        'nfce_id', // ID da NFCe quando convertida em venda
        'comprovante_nao_fiscal_id', // ID do comprovante quando convertida em venda não fiscal
        'local_id'
    ];

    protected $casts = [
        'data_consignacao' => 'datetime',
        'data_vencimento' => 'datetime',
        'valor_total' => 'decimal:2'
    ];

    // Relacionamentos
    public function empresa()
    {
        return $this->belongsTo(Empresa::class, 'empresa_id');
    }

    public function usuario()
    {
        return $this->belongsTo(User::class, 'usuario_id');
    }

    public function cliente()
    {
        return $this->belongsTo(Cliente::class, 'cliente_id');
    }

    public function funcionario()
    {
        return $this->belongsTo(Funcionario::class, 'funcionario_id');
    }

    public function itens()
    {
        return $this->hasMany(ItemConsignacao::class, 'consignacao_id');
    }

    public function nfce()
    {
        return $this->belongsTo(Nfce::class, 'nfce_id');
    }

    public function comprovanteNaoFiscal()
    {
        return $this->belongsTo(ComprovanteNaoFiscal::class, 'comprovante_nao_fiscal_id');
    }

    public function local()
    {
        return $this->belongsTo(Localizacao::class, 'local_id');
    }

    // Scopes
    public function scopeAtivas($query)
    {
        return $query->where('status', 'ativa');
    }

    public function scopeFinalizadas($query)
    {
        return $query->where('status', 'finalizada');
    }

    public function scopeCanceladas($query)
    {
        return $query->where('status', 'cancelada');
    }

    // Métodos
    public function getStatusTextAttribute()
    {
        $status = [
            'ativa' => 'Aberta',
            'finalizada' => 'Finalizada',
            'cancelada' => 'Cancelada'
        ];

        return $status[$this->status] ?? 'Desconhecido';
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'ativa' => 'success',
            'finalizada' => 'info',
            'cancelada' => 'danger'
        ];

        return $colors[$this->status] ?? 'secondary';
    }

    public function isAtiva()
    {
        return $this->status === 'ativa';
    }

    public function isFinalizada()
    {
        return $this->status === 'finalizada';
    }

    public function isCancelada()
    {
        return $this->status === 'cancelada';
    }

    public function podeSerEditada()
    {
        return $this->isAtiva();
    }

    public function podeSerFinalizada()
    {
        return $this->isAtiva();
    }

    public function podeSerCancelada()
    {
        return $this->isAtiva();
    }
} 