<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CheckDatabaseConnection
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure(\Illuminate\Http\Request): (\Illuminate\Http\Response|\Illuminate\Http\RedirectResponse)  $next
     * @return \Illuminate\Http\Response|\Illuminate\Http\RedirectResponse
     */
    public function handle(Request $request, Closure $next)
    {
        // Lista de rotas que não precisam verificar conexão com banco
        $excludedRoutes = [
            'login',
            'logout',
            'password.*',
            'register',
            'home',
            '/',
            'api/health',
            'health'
        ];

        // Verifica se a rota atual está na lista de exclusão
        $currentRoute = $request->route() ? $request->route()->getName() : $request->path();
        foreach ($excludedRoutes as $excludedRoute) {
            if (str_contains($currentRoute, $excludedRoute)) {
                return $next($request);
            }
        }

        try {
            // Tenta fazer uma consulta simples para verificar a conexão
            DB::connection()->getPdo();
            
            // Se chegou até aqui, a conexão está funcionando
            return $next($request);
            
        } catch (\Exception $e) {
            // Log do erro de conexão
            Log::critical('Middleware detectou erro de conexão com banco de dados: ' . $e->getMessage(), [
                'url' => $request->fullUrl(),
                'method' => $request->method(),
                'ip' => $request->ip(),
                'user_agent' => $request->userAgent()
            ]);

            // Se for uma requisição AJAX, retorna JSON
            if ($request->expectsJson()) {
                return response()->json([
                    'error' => 'Serviço indisponível',
                    'message' => 'O servidor está temporariamente indisponível. Tente novamente em alguns instantes.',
                    'retry_after' => 30
                ], 503);
            }

            // Para requisições normais, redireciona para a página de erro
            return response()->view('errors.500', [], 503);
        }
    }
} 