<?php

namespace App\Http\Controllers;

use App\Models\ConfiguracaoFinanceira;
use Illuminate\Http\Request;

class ConfiguracaoFinanceiraController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'verified']);
        // Adicione o middleware de empresa, se disponível
    }
    
    /**
     * Exibe a tela de configurações financeiras
     */
    public function index()
    {
        // Recuperar configuração da empresa atual
        $empresa_id = session('empresa_id', auth()->user()->empresa_id ?? 1);
        $configuracao = ConfiguracaoFinanceira::where('empresa_id', $empresa_id)->first();
        
        // Criar configuração padrão se não existir
        if (!$configuracao) {
            $configuracao = ConfiguracaoFinanceira::create([
                'empresa_id' => $empresa_id,
                'juros_diario' => 0.033, // 1% ao mês ÷ 30 dias
                'multa_atraso' => 2.0,   // 2% fixo por atraso
                'carencia_dias' => 0,    // Sem carência
                'prazo_padrao_dias' => 30
            ]);
        }
        
        return view('financeiro.configuracoes', compact('configuracao'));
    }
    
    /**
     * Atualiza as configurações financeiras
     */
    public function update(Request $request)
    {
        // Validar os dados
        $request->validate([
            'juros_diario' => 'required|numeric|min:0|max:10',
            'multa_atraso' => 'required|numeric|min:0|max:100',
            'carencia_dias' => 'required|integer|min:0|max:30',
            'prazo_padrao_dias' => 'required|integer|min:1|max:365',
            'notificar_vencimento' => 'boolean',
            'dias_notificacao_previa' => 'required|integer|min:0|max:30',
            'texto_padrao_recibo' => 'nullable|string|max:2000',
            'instrucoes_padrao_boleto' => 'nullable|string|max:2000'
        ]);
        
        try {
            // Recuperar configuração da empresa atual
            $empresa_id = session('empresa_id', auth()->user()->empresa_id ?? 1);
            $configuracao = ConfiguracaoFinanceira::where('empresa_id', $empresa_id)->first();
            
            // Criar configuração se não existir
            if (!$configuracao) {
                $configuracao = new ConfiguracaoFinanceira();
                $configuracao->empresa_id = $empresa_id;
            }
            
            // Atualizar campos
            $configuracao->juros_diario = $request->juros_diario;
            $configuracao->multa_atraso = $request->multa_atraso;
            $configuracao->carencia_dias = $request->carencia_dias;
            $configuracao->prazo_padrao_dias = $request->prazo_padrao_dias;
            $configuracao->notificar_vencimento = $request->has('notificar_vencimento');
            $configuracao->dias_notificacao_previa = $request->dias_notificacao_previa;
            $configuracao->texto_padrao_recibo = $request->texto_padrao_recibo;
            $configuracao->instrucoes_padrao_boleto = $request->instrucoes_padrao_boleto;
            
            $configuracao->save();
            
            // Log da operação
            __createLog(request()->empresa_id, 'Configurações Financeiras', 'atualização', 'Configurações financeiras atualizadas');
            
            session()->flash("flash_success", "Configurações financeiras atualizadas com sucesso.");
            
        } catch (\Exception $e) {
            __createLog(request()->empresa_id, 'Configurações Financeiras', 'erro', $e->getMessage());
            session()->flash("flash_error", "Erro ao atualizar as configurações financeiras: " . $e->getMessage());
        }
        
        return redirect()->route('financeiro.configuracoes');
    }

    /**
     * Retorna as configurações financeiras via API
     */
    public function getConfig()
    {
        try {
            $empresa_id = session('empresa_id', auth()->user()->empresa_id ?? 1);
            $configuracao = ConfiguracaoFinanceira::where('empresa_id', $empresa_id)->first();
            
            if (!$configuracao) {
                $configuracao = ConfiguracaoFinanceira::create([
                    'empresa_id' => $empresa_id,
                    'juros_diario' => 0.033, // 1% ao mês ÷ 30 dias
                    'multa_atraso' => 2.0,   // 2% fixo por atraso
                    'carencia_dias' => 0,    // Sem carência
                    'prazo_padrao_dias' => 30
                ]);
            }
            
            return response()->json($configuracao);
            
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }
} 