<?php

namespace App\Http\Controllers;

use App\Models\Caixa;
use App\Models\ContaEmpresa;
use App\Models\ContaPagar;
use App\Models\ContaReceber;
use App\Models\Empresa;
use App\Models\FaturaNfce;
use App\Models\ItemServicoNfce;
use App\Models\Localizacao;
use App\Models\Nfce;
use App\Models\Nfe;
use App\Models\OrdemServico;
use App\Models\SangriaCaixa;
use App\Models\SuprimentoCaixa;
use App\Models\User;
use App\Utils\ContaEmpresaUtil;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Dompdf\Dompdf;
use App\Models\ItemContaEmpresa;
use App\Models\Devolucao;
use App\Models\ItemDevolucao;

class CaixaController extends Controller
{
    protected $util;
    public function __construct(ContaEmpresaUtil $util){
        $this->util = $util;
    }

    public function index()
    {
        // Busca o caixa aberto mais recente para o usuário
        $item = Caixa::where('usuario_id', Auth::user()->id)
                ->where('status', 1)
                ->orderBy('created_at', 'desc')
                ->first();

        if ($item == null) {
            session()->flash('flash_warning', 'Não há caixa aberto no momento!');
            return redirect()->route('caixa.create');
        }
        $valor_abertura = $item->valor_abertura;

        $somaTiposPagamento = [];
        $contas = [];
        $nfce = Nfce::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $item->id)
            ->with('fatura')  // Carrega faturas com eager loading
            ->get();

        $nfe = Nfe::where('empresa_id',  request()->empresa_id)
            ->where('caixa_id', $item->id)
            ->where('tpNF', 1)
            ->where('orcamento', 0)
            ->with('fatura')  // Carrega faturas com eager loading
            ->get();

        $pagar = ContaPagar::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $ordens = OrdemServico::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)
        ->get();

        $data = $this->agrupaDados($nfce, $nfe, $ordens, null);

        // Criar versão expandida dos dados, com pagamentos separados
        $dadosExpandidos = $this->expandeDadosPorPagamento($data);

        // Agrupar os dados expandidos por venda
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendas += $valor;
            }
        }

        $somaTiposPagamento = $this->somaTiposPagamento($data);
        $contas = $this->agrupaContas($pagar, $receber);
        $somaTiposContas = $this->somaTiposContas($contas);

        $suprimentos = [];

        $somaServicos = ItemServicoNfce::join('nfces', 'nfces.id', '=', 'item_servico_nfces.nfce_id')
        ->where('nfces.empresa_id', request()->empresa_id)->where('nfces.caixa_id', $item->id)
        ->sum('sub_total');

        $sangrias = [];
        if ($item != null) {
            $suprimentos = SuprimentoCaixa::where('caixa_id', $item->id)->get();
            $sangrias = SangriaCaixa::where('caixa_id', $item->id)->get();

            // Log para debug das sangrias
            Log::info('Caixa #'.$item->id.' - Buscando sangrias:', [
                'caixa_id' => $item->id,
                'quantidade_sangrias' => count($sangrias),
                'sangrias' => $sangrias->toArray()
            ]);
        }

        // Calcular total de compras (NFes de entrada)
        $totalCompras = 0;
        $compras = Nfe::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $item->id)
            ->where('tpNF', 0) // NFe de entrada (compra)
            ->where('estado', '!=', 'cancelado')
            ->get();

        foreach($compras as $c) {
            $totalCompras += $c->total;
        }

        // Buscar devoluções feitas neste caixa
        $devolucoes = [];
        try {
            // Consulta base para devoluções (não trocas)
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('is_troca', false)
                ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

            // Executar a consulta
            $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();

            // Registrar no log a quantidade de devoluções encontradas
            Log::info('Caixa #'.$item->id.' - Quantidade de devoluções: '.count($devolucoes));
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar devoluções: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $devolucoes = collect([]);
        }

        // Buscar trocas feitas neste caixa (is_troca = true)
        $todasOperacoes = [];
        try {
            // Consulta base que inclui todas as operações (devoluções e trocas)
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('caixa_id', $item->id); // Filtrar apenas as operações deste caixa

            // Executar a consulta
            $todasOperacoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar todas as operações: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $todasOperacoes = collect([]);
        }

        // Calcular valores de devoluções por tipo de pagamento
        $devolucoesPorTipo = [
            'dinheiro' => 0, // tipo_pagamento = 01
            'cartao' => 0,    // tipo_pagamento = 03, 04
            'credito' => 0,   // tipo_pagamento = 00
            'pix' => 0        // tipo_pagamento = 17
        ];

        $totalDevolucoes = 0;
        $devolucoesDinheiroPix = 0; // <-- NOVA VARIÁVEL SIMPLIFICADA

        foreach($devolucoes as $devolucao) {
            $totalDevolucoes += $devolucao->valor_total;
            // Categorizar por tipo de pagamento
            if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                $devolucoesPorTipo['dinheiro'] += $devolucao->valor_total;
                $devolucoesDinheiroPix += $devolucao->valor_total;
            }
            elseif (in_array($devolucao->tipo_pagamento, ['03', '04'])) { // Cartões
                $devolucoesPorTipo['cartao'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '00') { // Crédito Loja
                $devolucoesPorTipo['credito'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '17') { // PIX
                $devolucoesPorTipo['pix'] += $devolucao->valor_total;
                $devolucoesDinheiroPix += $devolucao->valor_total;
            }
        }

        // Filtrar as trocas (is_troca = true) da coleção de todas as operações
        $trocas = $todasOperacoes->where('is_troca', true);

        // Calcular totais para trocas
        $trocas->totalEntradas = 0; // Total de diferenças pagas pelo cliente (valor negativo no BD)
        $trocas->totalSaidas = 0;   // Total de diferenças pagas ao cliente (valor positivo no BD)

        foreach ($trocas as $troca) {
            if ($troca->valor_diferenca < 0) {
                // Valor negativo significa que o cliente pagou (entrada)
                $trocas->totalEntradas += abs($troca->valor_diferenca);
            } else if ($troca->valor_diferenca > 0) {
                // Valor positivo significa que o cliente recebeu (saída)
                $trocas->totalSaidas += $troca->valor_diferenca;
            }
        }

        // Calcular totais de crediário, boleto e crédito loja
        $totalCrediario = $this->totalizaVendasCrediario($data);
        $totalBoleto = $this->totalizaVendasBoleto($data);
        $totalCreditoLoja = $this->totalizaVendasCreditoLoja($data);

        // Calcular valor em caixa
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        // Calcular total recebido de contas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        // Somar suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $item->id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $item->id)->sum('valor');

        // Process accounts receivable with multiple payment methods
        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        if ($item != null) {

            $contasEmpresa = ContaEmpresa::where('empresa_id', request()->empresa_id)->get();
            return view('caixa.index', compact(
                'item',
                'data',
                'dadosExpandidos',
                'vendasAgrupadas',
                'somaTiposPagamento',
                'valor_abertura',
                'somaServicos',
                'totalVendas',
                'totalCompras',
                'suprimentos',
                'sangrias',
                'contas',
                'somaTiposContas',
                'receber',
                'pagar',
                'contasEmpresa',
                'recebimentosExpandidos',
                'recebimentosAgrupados',
                'devolucoes',
                'devolucoesPorTipo',
                'totalDevolucoes',
                'trocas',
                'todasOperacoes',
                'valorEmCaixa'
            ));
        } else {
            session()->flash('flash_warning', 'Não há caixa aberto no momento!');
            return redirect()->back();
        }
    }

    private function agrupaDados($nfce, $nfe, $ordens, $compras)
    {
        $temp = [];
        foreach ($nfe as $v) {
            // Adiciona apenas NFes de venda (não compras)
            if(isset($v->tpNF) && $v->tpNF == 1) {
                $v->tipo = 'Pedido';
                $v->receita = 1;
                // Certificar que as faturas são carregadas para o cálculo correto
                if (!isset($v->fatura)) {
                    $v->load('fatura');
                }
                // Assegurar que o total reflete o valor com desconto aplicado
                if (isset($v->desconto) && $v->desconto > 0) {
                    // NFe já tem o desconto aplicado no campo total, então não precisamos subtrair novamente
                    $v->total_original = $v->total; // Para referência futura
                } else {
                    $v->total_original = $v->total;
                }
                // Garantir que temos os itens carregados para verificação do subtotal correto
                if (!isset($v->itens)) {
                    $v->load('itens');
                }
                // Recalcular o total com base nos itens, se necessário
                $subtotal_itens = 0;
                foreach ($v->itens as $item) {
                    $subtotal_itens += $item->sub_total;
                }
                // Log para diagnóstico
                Log::info('NFe #'.$v->id.' - Total: '.$v->total.', Subtotal Itens: '.$subtotal_itens);
                // Se houver uma discrepância significativa, usar o total calculado
                if (abs($v->total - $subtotal_itens) > 0.5 && !isset($v->desconto)) {
                    $v->total = $subtotal_itens;
                    Log::info('Ajustando total da NFe #'.$v->id.' para '.$v->total);
                }

                array_push($temp, $v);
            }
        }
        foreach ($nfce as $v) {
            $v->tipo = 'PDV';
            $v->receita = 1;
            if (!isset($v->fatura)) {
                $v->load('fatura');
            }
            if (!isset($v->itens)) {
                $v->load('itens');
            }
            // Calcular total bruto com base nos itens
            $valorProdutos = 0;
            foreach ($v->itens as $item) {
                $valorProdutos += $item->sub_total;
            }
            // Nunca sobrescrever o total se já estiver correto
            // Se for necessário ajustar, considerar desconto e acrescimo
            if (isset($v->desconto) && $v->desconto > 0) {
                $v->total_original = $valorProdutos;
                $descontoAplicado = $valorProdutos - $v->total;
                if (abs($descontoAplicado - $v->desconto) > 0.5) {
                    $v->total = $valorProdutos - $v->desconto + ($v->acrescimo ?? 0);
                    Log::info('Ajustando total da NFCe #'.$v->id.' para '.$v->total.' considerando desconto de R$'.$v->desconto.' e acrescimo de R$'.($v->acrescimo ?? 0));
                }
            } else if (isset($v->acrescimo) && $v->acrescimo > 0) {
                // Se só houver acrescimo
                if (abs(($valorProdutos + $v->acrescimo) - $v->total) > 0.5) {
                    $v->total = $valorProdutos + $v->acrescimo;
                    Log::info('Ajustando total da NFCe #'.$v->id.' para '.$v->total.' considerando apenas acrescimo de R$'.$v->acrescimo);
                }
            } else {
                if (abs($v->total - $valorProdutos) > 0.5) {
                    $v->total_original = $v->total;
                    $v->total = $valorProdutos;
                    Log::info('Ajustando total da NFCe #'.$v->id.' para '.$v->total.' com base no valor dos produtos');
                } else {
                    $v->total_original = $v->total;
                }
            }
            array_push($temp, $v);
        }

        if($ordens != null){
            foreach ($ordens as $v) {
                $v->tipo = 'OS';
                $v->receita = 1;
                array_push($temp, $v);
            }
        }

        // Removo o bloco que adicionava compras ao array de dados
        // As compras não devem aparecer na movimentação de caixa

        // Ordenar registros do mais recente para o mais antigo
        usort($temp, function($a, $b){
            return $a['created_at'] < $b['created_at'] ? 1 : -1;
        });

        // Limitar o número de registros para evitar consumo excessivo de memória
        // Exibir no máximo 100 registros mais recentes
        if (count($temp) > 100) {
            $temp = array_slice($temp, 0, 100);
        }

        return $temp;
    }

    private function agrupaContas($pagar, $receber)
    {
        $temp = [];
        foreach ($pagar as $c) {
            $c->tipo = 'Conta Paga';
            array_push($temp, $c);
        }
        foreach ($receber as $c) {
            $c->tipo = 'Conta Recebida';
            array_push($temp, $c);
        }
        return $temp;
    }

    /**
     * Calcula a soma por tipo de pagamento, considerando o novo formato de múltiplos pagamentos
     *
     * Agora consultamos diretamente a tabela de faturas para cada venda, onde todos os tipos
     * de pagamento são registrados separadamente. Esta abordagem é mais precisa e permite
     * o uso correto do sistema de múltiplos pagamentos em uma única venda.
     *
     * @param array $vendas Array de objetos de venda (NFCe, NFe, OS)
     * @return array Soma total por tipo de pagamento
     */
    private function somaTiposPagamento($vendas)
    {
        $tipos = $this->preparaTipos();

        foreach ($vendas as $v) {
            if ($v->estado != 'cancelado' && $v->receita == 1) {
                // Se for uma Ordem de Serviço (OS), trate diferentemente
                if ($v->tipo == 'OS') {
                    // Ordens de serviço têm forma_pagamento diretamente no objeto
                    if (isset($v->forma_pagamento) && isset($tipos[trim($v->forma_pagamento)])) {
                        $tipos[trim($v->forma_pagamento)] += $v->valor;
                    }
                }
                // Para os outros tipos de registros (PDV e Pedido), usar a fatura
                else {
                    // Carregar as faturas diretamente do banco se não estiverem no objeto
                    $faturas = $v->fatura ?? FaturaNfce::where('nfce_id', $v->id)->get();

                    if ($faturas && count($faturas) > 0) {
                        // Calcular a soma de todas as faturas
                        $totalFaturas = 0;
                        foreach ($faturas as $f) {
                            $totalFaturas += $f->valor;
                        }

                        // Verificar se existe discrepância entre o total das faturas e o total da venda
                        $fator = 1; // Por padrão, não ajustar
                        if (abs($totalFaturas - $v->total) > 0.5) {
                            // Se há diferença (devido a descontos), calculamos o fator de ajuste
                            // Verificação adicional para evitar divisão por zero
                            $fator = $totalFaturas > 0 ? $v->total / $totalFaturas : 0;
                            Log::info('Ajustando valores de faturas no somaTiposPagamento: Venda #'.$v->id.
                                     ' - Total: '.$v->total.
                                     ', Total Faturas: '.$totalFaturas.
                                     ', Fator: '.$fator);
                        }

                        // Adicionar valores ajustados por forma de pagamento
                        foreach ($faturas as $f) {
                            if (isset($tipos[trim($f->tipo_pagamento)])) {
                                // Aplicar o fator de ajuste para refletir descontos
                                $valorAjustado = round($f->valor * $fator, 2);
                                $tipos[trim($f->tipo_pagamento)] += $valorAjustado;

                                if ($fator != 1) {
                                    Log::info('Fatura tipo '.$f->tipo_pagamento.
                                             ' - Valor Original: '.$f->valor.
                                             ', Valor Ajustado: '.$valorAjustado);
                                }
                            }
                        }
                    }
                    // Caso não tenha faturas registradas, usar o tipo_pagamento padrão
                    else if (isset($v->tipo_pagamento) && isset($tipos[trim($v->tipo_pagamento)])) {
                        $tipos[trim($v->tipo_pagamento)] += $v->total;
                    }
                }
            }
        }
        return $tipos;
    }

    private function somaTiposContas($contas)
    {
        $tipos = $this->preparaTipos();

        foreach ($contas as $c) {
            if ($c->status == 1) {
                if(isset($tipos[trim($c->tipo_pagamento)])){
                    $tipos[trim($c->tipo_pagamento)] += $c->valor_integral;
                }
            }
        }
        return $tipos;
    }


    private function preparaTipos()
    {
        $temp = [];
        foreach (Nfce::tiposPagamento() as $key => $tp) {
            $temp[$key] = 0;
        }
        return $temp;
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $item = Caixa::where('usuario_id', Auth::user()->id)->where('status', 1)->first();

        return view('caixa.create', compact('item'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            // Usar uma transação para garantir atomicidade da operação
            return DB::transaction(function() use ($request) {
                // Verificar se já existe um caixa aberto para este usuário - com bloqueio para evitar condições de corrida
                $existingCaixa = Caixa::where('usuario_id', Auth::user()->id)
                    ->where('status', 1)
                    ->lockForUpdate()  // Adiciona bloqueio para evitar condições de corrida
                    ->first();

                if ($existingCaixa) {
                    // Se já existe um caixa aberto, avisar o usuário e redirecionar
                    session()->flash('flash_warning', 'Você já possui um caixa aberto! Não é possível abrir mais de um caixa ao mesmo tempo.');
                    return redirect()->route('caixa.index');
                }

                // Continuar com a criação do caixa, se não existir um aberto
                $request->merge([
                    'usuario_id' => Auth::user()->id,
                    'valor_abertura' => __convert_value_bd($request->valor_abertura),
                    'observacao' => $request->observacao ?? '',
                    'status' => 1,
                    'valor_fechamento' => 0,
                ]);
                $item = Caixa::create($request->all());

                $descricaoLog = $item->usuario->name . " | CAIXA ABERTO - abertura: " . __data_pt($item->created_at) . " - valor abertura: " . __moeda($item->valor_abertura);
                __createLog($request->empresa_id, 'Caixa', 'cadastrar', $descricaoLog);
                session()->flash('flash_success', 'Caixa aberto com sucesso!');

                return redirect()->route('caixa.index');
            });
        } catch (\Exception $e) {
            __createLog($request->empresa_id, 'Caixa', 'erro', $e->getMessage());
            session()->flash('flash_error', 'Não foi possível abrir o caixa: ' . $e->getMessage());
            return redirect()->route('caixa.index');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $item = Caixa::findOrFail($id);
        $valor_abertura = $item->valor_abertura;

        $nfce = Nfce::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $nfe = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)
        ->where('tpNF', 1)
        ->where('orcamento', 0)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $pagar = ContaPagar::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();

        $data = $this->agrupaDados($nfce, $nfe, null, null);

        // Criar versão expandida dos dados, com pagamentos separados
        $dadosExpandidos = $this->expandeDadosPorPagamento($data);

        // Agrupar os dados expandidos por venda
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        // Processar contas a receber com múltiplas formas de pagamento
        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $totalVendas += $v->total;
            }
        }

        $somaTiposPagamento = $this->somaTiposPagamento($data);
        $contas = $this->agrupaContas($pagar, $receber);
        $somaTiposContas = $this->somaTiposContas($contas);

        $suprimentos = SuprimentoCaixa::where('caixa_id', $item->id)->get();
        $sangrias = SangriaCaixa::where('caixa_id', $item->id)->get();

        // Calcular soma total de suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $item->id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $item->id)->sum('valor');

        // Calcular total de serviços
        $somaServicos = ItemServicoNfce::join('nfces', 'nfces.id', '=', 'item_servico_nfces.nfce_id')
            ->where('nfces.empresa_id', request()->empresa_id)
            ->where('nfces.caixa_id', $item->id)
            ->sum('sub_total');

        // Calcular total de compras (NFes de entrada)
        $totalCompras = 0;
        $compras = Nfe::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $item->id)
            ->where('tpNF', 0) // NFe de entrada (compra)
            ->where('estado', '!=', 'cancelado')
            ->get();

        foreach($compras as $c) {
            $totalCompras += $c->total;
        }

        // Buscar devoluções feitas neste caixa
        $devolucoes = [];
        try {
            // Consulta base
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('is_troca', false)
                ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

            // Executar a consulta
            $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar devoluções: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $devolucoes = collect([]);
        }

        // Calcular valores de devoluções por tipo de pagamento
        $devolucoesPorTipo = [
            'dinheiro' => 0, // tipo_pagamento = 01
            'cartao' => 0,    // tipo_pagamento = 03, 04
            'credito' => 0,   // tipo_pagamento = 00
            'pix' => 0        // tipo_pagamento = 17
        ];

        $totalDevolucoes = 0;

        foreach($devolucoes as $devolucao) {
            $totalDevolucoes += $devolucao->valor_total;

            // Categorizar por tipo de pagamento
            if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                $devolucoesPorTipo['dinheiro'] += $devolucao->valor_total;
            }
            elseif (in_array($devolucao->tipo_pagamento, ['03', '04'])) { // Cartões
                $devolucoesPorTipo['cartao'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '00') { // Crédito Loja
                $devolucoesPorTipo['credito'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '17') { // PIX
                $devolucoesPorTipo['pix'] += $devolucao->valor_total;
            }
        }

        // Ajustar o cálculo do valor final considerando as devoluções em dinheiro e PIX
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        // Subtrair as devoluções em dinheiro e PIX do valor em caixa
        $valorEmCaixa -= ($devolucoesPorTipo['dinheiro'] + $devolucoesPorTipo['pix']);

        // Buscar trocas realizadas no caixa
        $trocas = $this->getTrocasCaixa($item->id);

        // Buscar todas as operações para processamento
        $todasOperacoes = collect([]);
        try {
            // Consulta base que inclui todas as operações (devoluções e trocas)
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('caixa_id', $item->id); // Filtrar apenas as operações deste caixa

            // Executar a consulta
            $todasOperacoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar todas as operações: ' . $e->getMessage());
        }

        // Calcular total de contas recebidas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            // Se tiver valor_recebido definido e for maior que zero, usar esse valor
            // Caso contrário, usar o valor_integral
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        // Calcular soma da receita e despesa para fórmula
        $somaReceita = 0;
        $somaDespesa = 0;

        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;

                // Verificar se é venda à vista
                $receita = true;
                if ($v->tipo_pagamento == '05' || $v->tipo_pagamento == '06' ||
                    $v->tipo_pagamento == '15' || $v->tipo_pagamento == '16') {
                    $receita = false; // Não é receita à vista
                }

                if ($receita) {
                    $somaReceita += $valor;
                }
            }
        }

        $contasEmpresa = ContaEmpresa::where('empresa_id', request()->empresa_id)->get();

        return view('caixa.show', compact(
            'item',
            'data',
            'dadosExpandidos',
            'vendasAgrupadas',
            'somaTiposPagamento',
            'valor_abertura',
            'somaServicos',
            'totalVendas',
            'totalCompras',
            'suprimentos',
            'sangrias',
            'contas',
            'somaTiposContas',
            'receber',
            'pagar',
            'contasEmpresa',
            'recebimentosExpandidos',
            'recebimentosAgrupados',
            'devolucoes',
            'devolucoesPorTipo',
            'totalDevolucoes',
            'trocas',
            'todasOperacoes',
            'valorRecebidoTotal',
            'somaReceita',
            'somaDespesa',
            'somaSuprimento',
            'somaSangria'
        ));
    }

    public function imprimir80($id)
    {
        $item = Caixa::findOrFail($id);
        $valor_abertura = $item->valor_abertura;
        $empresa = Empresa::findOrFail(request()->empresa_id);
        $usuario = User::findOrFail($item->usuario_id);
        $nfce = Nfce::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();

        $nfe = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)->where('tpNF', 1)
        ->where('orcamento', 0)->get();
        $ordens = OrdemServico::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)->get();

        // Não queremos mais incluir compras
        // $compras = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)->where('tpNF', 0)
        // ->where('orcamento', 0)
        // ->get();

        // Passamos null para compras em agrupaDados
        $data = $this->agrupaDados($nfce, $nfe, $ordens, null);
        $somaTiposPagamento = $this->somaTiposPagamento($data);

        $data_abertura = date('d/m/Y', strtotime($item->created_at));
        $abertura_hora = date('H:i:s', strtotime($item->created_at));

        $data_fechamento = $item->status == 1 ? 'Em aberto' : date('d/m/Y', strtotime($item->updated_at));
        $fechamento_hora = $item->status == 1 ? '--:--:--' : date('H:i:s', strtotime($item->updated_at));

        $suprimentos = [];
        $sangrias = [];
        $contas = [];

        $pagar = ContaPagar::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();

        $contas = $this->agrupaContas($pagar, $receber);
        $somaTiposContas = $this->somaTiposContas($contas);

        if ($item != null) {
            $suprimentos = SuprimentoCaixa::where('caixa_id', $item->id)->get();
            $sangrias = SangriaCaixa::where('caixa_id', $item->id)->get();
        }

        $somaServicos = ItemServicoNfce::join('nfces', 'nfces.id', '=', 'item_servico_nfces.nfce_id')
        ->where('nfces.empresa_id', request()->empresa_id)->where('nfces.caixa_id', $item->id)
        ->sum('sub_total');

        // Buscar devoluções feitas neste caixa
        $devolucoes = [];
        try {
            // Consulta base
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('is_troca', false)
                ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

            // Executar a consulta
            $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar devoluções: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $devolucoes = collect([]);
        }

        // Calcular valores de devoluções por tipo de pagamento
        $devolucoesPorTipo = [
            'dinheiro' => 0, // tipo_pagamento = 01
            'cartao' => 0,    // tipo_pagamento = 03, 04
            'credito' => 0,   // tipo_pagamento = 00
            'pix' => 0        // tipo_pagamento = 17
        ];

        $totalDevolucoes = 0;

        foreach($devolucoes as $devolucao) {
            $totalDevolucoes += $devolucao->valor_total;

            // Categorizar por tipo de pagamento
            if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                $devolucoesPorTipo['dinheiro'] += $devolucao->valor_total;
            }
            elseif (in_array($devolucao->tipo_pagamento, ['03', '04'])) { // Cartões
                $devolucoesPorTipo['cartao'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '00') { // Crédito Loja
                $devolucoesPorTipo['credito'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '17') { // PIX
                $devolucoesPorTipo['pix'] += $devolucao->valor_total;
            }
        }

        // Ajustar o cálculo do valor final considerando as devoluções em dinheiro e PIX
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        // Subtrair as devoluções em dinheiro e PIX do valor em caixa
        $valorEmCaixa -= ($devolucoesPorTipo['dinheiro'] + $devolucoesPorTipo['pix']);

        // Buscar trocas realizadas no caixa
        $trocas = $this->getTrocasCaixa($item->id);

        $produtos = $this->totalizaProdutos($data);
        $dadosExpandidos = $this->expandeDadosPorPagamento($data);
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        // Calcular o total das vendas
        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendas += $valor;
            }
        }

        // Calcular vendas a crediário, boleto e crédito loja
        $totalCrediario = $this->totalizaVendasCrediario($data);
        $totalBoleto = $this->totalizaVendasBoleto($data);
        $totalCreditoLoja = $this->totalizaVendasCreditoLoja($data);

        // Calcular vendas recebidas (vendas efetivas)
        $totalVendasEfetivas = $totalVendas - $totalCrediario - $totalBoleto - $totalCreditoLoja;

        // Buscar suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $item->id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $item->id)->sum('valor');

        // Calcular total de contas recebidas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            // Se tiver valor_recebido definido e for maior que zero, usar esse valor
            // Caso contrário, usar o valor_integral
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        $dompdf = new Dompdf([
            'enable_remote' => true,
            'chroot' => public_path(),
            'font_height_ratio' => 0.9, // Ajustar altura da fonte para impressão térmica
            'enable_css_float' => true,
            'enable_html5_parser' => true
        ]);
        $dompdf->loadHtml(view('caixa.imprimir_80', compact(
            'item',
            'data',
            'dadosExpandidos',
            'vendasAgrupadas',
            'somaTiposPagamento',
            'valor_abertura',
            'valorEmCaixa',
            'empresa', // use empresa aqui, para manter compatibilidade com a view
            'usuario',
            'data_abertura',
            'abertura_hora',
            'data_fechamento',
            'fechamento_hora',
            'suprimentos',
            'sangrias',
            'somaServicos',
            'contas',
            'somaTiposContas',
            'devolucoes',
            'devolucoesPorTipo',
            'totalDevolucoes',
            'valorEmCaixa',
            'produtos',
            'recebimentosExpandidos',
            'recebimentosAgrupados',
            'totalVendas',
            'totalCrediario',
            'totalBoleto',
            'totalCreditoLoja',
            'totalVendasEfetivas',
            'somaSuprimento',
            'somaSangria',
            'valorRecebidoTotal',
            'pagar',
            'trocas'
        ))->with('config', $empresa)->render());

        // Configurar tamanho de papel para 80mm (aproximadamente 3.14 polegadas de largura)
        // O DomPDF usa pontos como unidade (72 pontos = 1 polegada)
        // 80mm = aproximadamente 226.8 pontos (3.14 * 72)
        // Deixamos a altura como 0 para ajustar automaticamente ao conteúdo
        $customPaper = array(0, 0, 226.8, 800);
        $dompdf->setPaper($customPaper, 'portrait');

        $dompdf->render();

        // Configurar o nome do arquivo para refletir que é um relatório de 80mm
        $dompdf->stream("Relatório de caixa 80mm - #" . $id . ".pdf", array("Attachment" => false));
    }

    /**
     * Gera o relatório de caixa em formato PDF (A4)
     *
     * @param int $id ID do caixa
     * @return \Illuminate\Http\Response
     */
    public function imprimir($id)
    {
        $item = Caixa::findOrFail($id);
        $valor_abertura = $item->valor_abertura;

        $nfce = Nfce::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $nfe = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)
        ->where('tpNF', 1)
        ->where('orcamento', 0)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $pagar = ContaPagar::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();

        $data = $this->agrupaDados($nfce, $nfe, null, null);

        // Criar versão expandida dos dados, com pagamentos separados
        $dadosExpandidos = $this->expandeDadosPorPagamento($data);

        // Agrupar os dados expandidos por venda
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        // Processar contas a receber com múltiplas formas de pagamento
        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $totalVendas += $v->total;
            }
        }

        $somaTiposPagamento = $this->somaTiposPagamento($data);
        $contas = $this->agrupaContas($pagar, $receber);
        $somaTiposContas = $this->somaTiposContas($contas);

        $suprimentos = SuprimentoCaixa::where('caixa_id', $item->id)->get();
        $sangrias = SangriaCaixa::where('caixa_id', $item->id)->get();

        // Calcular total de serviços
        $somaServicos = ItemServicoNfce::join('nfces', 'nfces.id', '=', 'item_servico_nfces.nfce_id')
            ->where('nfces.empresa_id', request()->empresa_id)
            ->where('nfces.caixa_id', $item->id)
            ->sum('sub_total');

        // Calcular total de compras (NFes de entrada)
        $totalCompras = 0;
        $compras = Nfe::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $item->id)
            ->where('tpNF', 0) // NFe de entrada (compra)
            ->where('estado', '!=', 'cancelado')
            ->get();

        foreach($compras as $c) {
            $totalCompras += $c->total;
        }

        // Buscar devoluções feitas neste caixa
        $devolucoes = [];
        try {
            // Consulta base
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('is_troca', false)
                ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

            // Executar a consulta
            $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar devoluções: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $devolucoes = collect([]);
        }

        // Calcular valores de devoluções por tipo de pagamento
        $devolucoesPorTipo = [
            'dinheiro' => 0, // tipo_pagamento = 01
            'cartao' => 0,    // tipo_pagamento = 03, 04
            'credito' => 0,   // tipo_pagamento = 00
            'pix' => 0        // tipo_pagamento = 17
        ];

        $totalDevolucoes = 0;

        foreach($devolucoes as $devolucao) {
            $totalDevolucoes += $devolucao->valor_total;

            // Categorizar por tipo de pagamento
            if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                $devolucoesPorTipo['dinheiro'] += $devolucao->valor_total;
            }
            elseif (in_array($devolucao->tipo_pagamento, ['03', '04'])) { // Cartões
                $devolucoesPorTipo['cartao'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '00') { // Crédito Loja
                $devolucoesPorTipo['credito'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '17') { // PIX
                $devolucoesPorTipo['pix'] += $devolucao->valor_total;
            }
        }

        // Ajustar o cálculo do valor final considerando as devoluções em dinheiro e PIX
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        // Subtrair as devoluções em dinheiro e PIX do valor em caixa
        $valorEmCaixa -= ($devolucoesPorTipo['dinheiro'] + $devolucoesPorTipo['pix']);

        // Buscar trocas realizadas no caixa
        $trocas = $this->getTrocasCaixa($item->id);

        $data_hoje = date('d/m/Y');
        $produtos = $this->totalizaProdutos($data);

        $dadosExpandidos = $this->expandeDadosPorPagamento($data);
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        // Calcular o total das vendas
        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendas += $valor;
            }
        }

        // Calcular vendas a crediário, boleto e crédito loja
        $totalCrediario = $this->totalizaVendasCrediario($data);
        $totalBoleto = $this->totalizaVendasBoleto($data);
        $totalCreditoLoja = $this->totalizaVendasCreditoLoja($data);

        // Calcular vendas recebidas (vendas efetivas)
        $totalVendasEfetivas = $totalVendas - $totalCrediario - $totalBoleto - $totalCreditoLoja;

        // Buscar suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $item->id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $item->id)->sum('valor');

        // Calcular total de contas recebidas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            // Se tiver valor_recebido definido e for maior que zero, usar esse valor
            // Caso contrário, usar o valor_integral
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        // Calcular vendas canceladas
        $totalVendasCanceladas = 0;
        foreach($data as $v) {
            if($v->estado == 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendasCanceladas += $valor;
            }
        }

        // Adicionar o título para o relatório
        $title = "Relatório de Caixa #" . $id;

        // Buscar a empresa atual
        $empresa = Empresa::findOrFail(request()->empresa_id);

        // Adicionar informações de usuário, data de abertura e fechamento
        $usuario = User::findOrFail($item->usuario_id);
        $data_abertura = date('d/m/Y', strtotime($item->created_at));
        $abertura_hora = date('H:i:s', strtotime($item->created_at));

        $data_fechamento = $item->status == 1 ? 'Em aberto' : date('d/m/Y', strtotime($item->updated_at));
        $fechamento_hora = $item->status == 1 ? '--:--:--' : date('H:i:s', strtotime($item->updated_at));

        $dompdf = new Dompdf();
        $dompdf->loadHtml(view('caixa.imprimir', compact(
            'item',
            'data',
            'dadosExpandidos',
            'vendasAgrupadas',
            'somaTiposPagamento',
            'valor_abertura',
            'valorEmCaixa',
            'usuario',
            'data_abertura',
            'abertura_hora',
            'data_fechamento',
            'fechamento_hora',
            'suprimentos',
            'sangrias',
            'somaServicos',
            'contas',
            'somaTiposContas',
            'devolucoes',
            'devolucoesPorTipo',
            'totalDevolucoes',
            'data_hoje',
            'produtos',
            'recebimentosExpandidos',
            'recebimentosAgrupados',
            'totalVendas',
            'totalVendasCanceladas',
            'totalCrediario',
            'totalBoleto',
            'totalCreditoLoja',
            'totalVendasEfetivas',
            'somaSuprimento',
            'somaSangria',
            'valorRecebidoTotal',
            'pagar',
            'title',
            'trocas'
        ))->with('config', $empresa)->render());
        $dompdf->setPaper('A4', 'landscape');
        $dompdf->render();
        $dompdf->stream("Relatório de caixa.pdf", array("Attachment" => false));
    }

    private function totalizaProdutos($vendas){
        $produtos = [];
        $produtos_id = [];
        foreach($vendas as $v){
            foreach($v->itens as $item){
                if(!in_array($item->produto_id, $produtos_id)){
                    $quantidade = $item->quantidade;
                    if($item->produto->unidade == 'UN' || $item->produto->unidade == 'UNID'){
                        $quantidade = number_format($item->quantidade, 0);
                    }
                    $p = [
                        'id' => $item->produto->id,
                        'nome' => $item->produto->nome,
                        'quantidade' => $quantidade,
                        'valor_venda' => $item->produto->valor_unitario,
                        'valor_compra' => $item->produto->valor_compra
                    ];
                    array_push($produtos, $p);
                    array_push($produtos_id, $item->produto_id);
                }else{
                    //atualiza
                    for($i=0; $i<sizeof($produtos); $i++){
                        if($produtos[$i]['id'] == $item->produto_id){
                            $produtos[$i]['quantidade'] += $item->quantidade;

                            if($item->produto->unidade == 'UN' || $item->produto->unidade == 'UNID'){
                                $produtos[$i]['quantidade'] = number_format($produtos[$i]['quantidade'], 0);
                            }else{
                                $produtos[$i]['quantidade'] = number_format($produtos[$i]['quantidade'], 2);
                            }
                        }
                    }
                }
            }
        }

        return $produtos;
    }

    /**
     * Expande a lista de vendas, mostrando cada pagamento separadamente
     *
     * @param array $vendas Array de objetos de venda (NFCe, NFe, OS)
     * @return array Array expandido com pagamentos separados
     */
    private function expandeDadosPorPagamento($vendas)
    {
        $expandido = [];

        foreach ($vendas as $v) {
            // Se for uma Ordem de Serviço (OS), mantém a forma original
            if ($v->tipo == 'OS') {
                $v->doc_id = $v->id; // Referência ao ID original
                $v->pagamento_nome = isset($v->forma_pagamento) ?
                    Nfce::getTipoPagamento($v->forma_pagamento) : 'Não definido';
                $v->valor_pagamento = $v->valor;
                $v->a_prazo = in_array($v->forma_pagamento, ['05', '06', '15', '16']); // Crédito loja, crediário, boleto, depósito

                // Adicionar informações de parcelamento se estiverem disponíveis
                if(isset($v->num_parcelas) && $v->num_parcelas > 1) {
                    $v->parcelado = true;
                    $v->num_parcelas = $v->num_parcelas;
                    $v->valor_parcela = $v->valor / $v->num_parcelas;
                } else {
                    $v->parcelado = false;
                }

                array_push($expandido, $v);
            }
            // Para NFCe e NFe, expandir com base nas faturas
            else {
                // Carregar as faturas se não estiverem carregadas
                $faturas = $v->fatura ?? FaturaNfce::where('nfce_id', $v->id)->get();

                // Se tiver faturas, criar uma entrada para cada fatura
                if ($faturas && count($faturas) > 0) {
                    $totalFaturas = 0;

                    foreach ($faturas as $f) {
                        // Cria uma cópia do objeto original
                        $pagamento = clone $v;
                        $pagamento->pagamento_id = $f->id;
                        $pagamento->tipo_pagamento = $f->tipo_pagamento;
                        $pagamento->pagamento_nome = Nfce::getTipoPagamento($f->tipo_pagamento);

                        // Se o total das faturas for diferente do total (possível em caso de desconto),
                        // ajustamos proporcionalmente o valor da fatura
                        $pagamento->valor_pagamento = $f->valor;
                        $totalFaturas += $f->valor;

                        $pagamento->doc_id = $v->id; // Referência ao ID original
                        $pagamento->a_prazo = in_array($f->tipo_pagamento, ['05', '06', '15', '16']); // Crédito loja, crediário, boleto, depósito

                        // Adicionar informações de parcelamento se estiverem disponíveis
                        if(isset($f->num_parcelas) && $f->num_parcelas > 1) {
                            $pagamento->parcelado = true;
                            $pagamento->num_parcelas = $f->num_parcelas;
                            // Verificação adicional para evitar divisão por zero
                            $pagamento->valor_parcela = $f->num_parcelas > 0 ? $f->valor / $f->num_parcelas : 0;
                        } else {
                            $pagamento->parcelado = false;
                        }

                        array_push($expandido, $pagamento);
                    }

                    // Se existe uma diferença entre o total das faturas e o total com desconto,
                    // ajustamos proporcionalmente os valores
                    if (abs($totalFaturas - $v->total) > 0.5 && count($expandido) > 0) {
                        // Verificação adicional para evitar divisão por zero
                        $fator = $totalFaturas > 0 ? $v->total / $totalFaturas : 0;

                        // Log para diagnóstico
                        Log::info('Ajustando valores de faturas: NFCe #'.$v->id.
                               ' - Total: '.$v->total.
                               ', Total Faturas: '.$totalFaturas.
                               ', Fator: '.$fator);

                        // Encontrar todas as entradas desta NFCe e ajustar proporcionalmente
                        for ($i = count($expandido) - count($faturas); $i < count($expandido); $i++) {
                            if (isset($expandido[$i]) && $expandido[$i]->doc_id == $v->id) {
                                $valorOriginal = $expandido[$i]->valor_pagamento;
                                $expandido[$i]->valor_pagamento = round($valorOriginal * $fator, 2);

                                Log::info('Fatura #'.$expandido[$i]->pagamento_id.
                                       ' - Valor Original: '.$valorOriginal.
                                       ', Valor Ajustado: '.$expandido[$i]->valor_pagamento);
                            }
                        }
                    }
                }
                // Se não tiver faturas, usar o tipo_pagamento padrão
                else if (isset($v->tipo_pagamento)) {
                    $v->pagamento_nome = Nfce::getTipoPagamento($v->tipo_pagamento);
                    $v->valor_pagamento = $v->total;
                    $v->doc_id = $v->id; // Referência ao ID original
                    $v->a_prazo = in_array($v->tipo_pagamento, ['05', '06', '15', '16']); // Crédito loja, crediário, boleto, depósito

                    // Adicionar informações de parcelamento se estiverem disponíveis
                    if(isset($v->num_parcelas) && $v->num_parcelas > 1) {
                        $v->parcelado = true;
                        $v->num_parcelas = $v->num_parcelas;
                        // Verificação adicional para evitar divisão por zero
                        $v->valor_parcela = $v->num_parcelas > 0 ? $v->total / $v->num_parcelas : 0;
                    } else {
                        $v->parcelado = false;
                    }

                    array_push($expandido, $v);
                }
                // Se não tiver nem faturas nem tipo_pagamento, manter original
                else {
                    $v->pagamento_nome = 'Não definido';
                    $v->valor_pagamento = $v->total;
                    $v->doc_id = $v->id; // Referência ao ID original
                    $v->a_prazo = false;
                    $v->parcelado = false;
                    array_push($expandido, $v);
                }
            }
        }

        // Ordenar registros do mais recente para o mais antigo
        usort($expandido, function($a, $b){
            return $a->created_at < $b->created_at ? 1 : -1;
        });

        return $expandido;
    }

    /**
     * Agrupa os registros de pagamento por venda, mantendo todos os métodos de pagamento juntos
     * Este método é usado para exibir todas as formas de pagamento de uma venda em uma única linha
     *
     * @param array $dados Array de objetos expandidos por pagamento
     * @return array Array agrupado por ID da venda
     */
    private function agrupaVendas($dados)
    {
        $agrupados = [];

        // Agrupar por doc_id (ID da venda)
        foreach($dados as $item) {
            if(!isset($item->estado) || $item->estado != 'cancelado') {
                $docId = $item->doc_id;

                if(!isset($agrupados[$docId])) {
                    // Inicializa o registro da venda com os dados básicos
                    $agrupados[$docId] = (object)[
                        'doc_id' => $docId,
                        'tipo' => $item->tipo,
                        'cliente' => $item->cliente ?? null,
                        'created_at' => $item->created_at,
                        'receita' => $item->receita,
                        'estado' => $item->estado ?? null,
                        'pagamentos' => [],
                        'valor_total' => 0,
                        'tem_a_prazo' => false
                    ];
                }

                // Adiciona o pagamento ao array de pagamentos desta venda
                $pagamento = (object)[
                    'tipo_pagamento' => $item->tipo_pagamento ?? null,
                    'pagamento_nome' => $item->pagamento_nome ?? 'Não definido',
                    'valor_pagamento' => $item->valor_pagamento,
                    'a_prazo' => $item->a_prazo ?? false,
                    'parcelado' => $item->parcelado ?? false
                ];

                // Adicionar informações de parcelamento se disponíveis
                if(isset($item->parcelado) && $item->parcelado) {
                    $pagamento->num_parcelas = $item->num_parcelas;
                    $pagamento->valor_parcela = $item->valor_parcela;
                }

                $agrupados[$docId]->pagamentos[] = $pagamento;
                $agrupados[$docId]->valor_total += $item->valor_pagamento;

                // Se qualquer pagamento for a prazo, marca a venda como tendo pagamento a prazo
                if($item->a_prazo) {
                    $agrupados[$docId]->tem_a_prazo = true;
                }
            }
        }

        // Converte array associativo para array indexado
        $resultado = [];
        foreach($agrupados as $venda) {
            $resultado[] = $venda;
        }

        // Ordenar registros do mais recente para o mais antigo
        usort($resultado, function($a, $b){
            return $a->created_at < $b->created_at ? 1 : -1;
        });

        return $resultado;
    }

    /**
     * Expande os dados das contas a receber incluindo todas as formas de pagamento
     * Similar à função expandeDadosPorPagamento para vendas
     *
     * @param array $receber Array de objetos ContaReceber
     * @return array Array expandido com pagamentos separados
     */
    private function expandeContasReceberPorPagamento($receber)
    {
        $expandido = [];

        foreach ($receber as $c) {
            // Carregar as faturas relacionadas
            if (!isset($c->faturas)) {
                $c->load('faturas');
            }

            // Se tem faturas, criar uma entrada para cada fatura
            if ($c->faturas && count($c->faturas) > 0) {
                foreach ($c->faturas as $f) {
                    // Cria uma cópia do objeto original
                    $pagamento = clone $c;
                    $pagamento->pagamento_id = $f->id;
                    $pagamento->tipo_pagamento = $f->tipo_pagamento;
                    $pagamento->pagamento_nome = ContaReceber::tiposPagamento()[$f->tipo_pagamento] ?? 'Não definido';
                    $pagamento->valor_pagamento = $f->valor;
                    $pagamento->doc_id = $c->id; // Referência ao ID original
                    $pagamento->valor_integral_original = $c->valor_integral; // Valor total da conta

                    array_push($expandido, $pagamento);
                }
            }
            // Se não tiver faturas, usar o tipo_pagamento padrão
            else if (isset($c->tipo_pagamento)) {
                $c->pagamento_nome = ContaReceber::tiposPagamento()[$c->tipo_pagamento] ?? 'Não definido';
                $c->valor_pagamento = $c->valor_recebido > 0 ? $c->valor_recebido : $c->valor_integral;
                $c->doc_id = $c->id; // Referência ao ID original
                $c->valor_integral_original = $c->valor_integral; // Valor total da conta

                array_push($expandido, $c);
            }
            // Se não tiver nem faturas nem tipo_pagamento, manter original
            else {
                $c->pagamento_nome = 'Não definido';
                $c->valor_pagamento = $c->valor_recebido > 0 ? $c->valor_recebido : $c->valor_integral;
                $c->doc_id = $c->id; // Referência ao ID original
                $c->valor_integral_original = $c->valor_integral; // Valor total da conta

                array_push($expandido, $c);
            }
        }

        // Ordenar registros do mais recente para o mais antigo
        usort($expandido, function($a, $b){
            return $a->created_at < $b->created_at ? 1 : -1;
        });

        return $expandido;
    }

    /**
     * Agrupa os registros de pagamento das contas por conta, mantendo todos os métodos de pagamento juntos
     *
     * @param array $dados Array de objetos expandidos por pagamento
     * @return array Array agrupado por ID da conta
     */
    private function agrupaContasReceber($dados)
    {
        $agrupados = [];

        // Agrupar por doc_id (ID da conta)
        foreach($dados as $item) {
            $docId = $item->doc_id;

            if(!isset($agrupados[$docId])) {
                // Inicializa o registro da conta com os dados básicos
                $agrupados[$docId] = (object)[
                    'doc_id' => $docId,
                    'tipo' => $item->tipo,
                    'cliente' => $item->cliente ?? null,
                    'descricao' => $item->descricao ?? $item->referencia ?? 'Sem descrição',
                    'created_at' => $item->created_at,
                    'status' => $item->status,
                    'pagamentos' => [],
                    'valor_total_pago' => 0,
                    'valor_integral' => $item->valor_integral_original
                ];
            }

            // Adiciona o pagamento ao array de pagamentos desta conta
            $pagamento = (object)[
                'tipo_pagamento' => $item->tipo_pagamento ?? null,
                'pagamento_nome' => $item->pagamento_nome ?? 'Não definido',
                'valor_pagamento' => $item->valor_pagamento ?? 0,
            ];

            $agrupados[$docId]->pagamentos[] = $pagamento;
            $agrupados[$docId]->valor_total_pago += $item->valor_pagamento ?? 0;
        }

        // Converte array associativo para array indexado
        $resultado = [];
        foreach($agrupados as $conta) {
            $resultado[] = $conta;
        }

        // Ordenar registros do mais recente para o mais antigo
        usort($resultado, function($a, $b){
            return $a->created_at < $b->created_at ? 1 : -1;
        });

        return $resultado;
    }

    /**
     * Calcula o total de vendas no crediário
     *
     * @param array $vendas Array de objetos de venda (NFCe, NFe, OS)
     * @return float Total de vendas no crediário
     */
    private function totalizaVendasCrediario($vendas)
    {
        $total = 0;
        foreach ($vendas as $v) {
            if ($v->estado != 'cancelado') {
                // Se for OS com pagamento crediário
                if ($v->tipo == 'OS' && isset($v->forma_pagamento) && $v->forma_pagamento == '06') {
                    $total += $v->valor;
                }
                // Se for NFCe ou NFe, verificar faturas
                else if ($v->tipo != 'OS') {
                    // Carregar faturas se necessário
                    $faturas = $v->fatura ?? FaturaNfce::where('nfce_id', $v->id)->get();

                    // Somar valores de faturas com tipo_pagamento = '06' (crediário)
                    if ($faturas && count($faturas) > 0) {
                        foreach ($faturas as $f) {
                            if ($f->tipo_pagamento == '06') {
                                $total += $f->valor;
                            }
                        }
                    }
                    // Se não tiver faturas mas tiver tipo_pagamento = '06', considerar o total
                    else if (isset($v->tipo_pagamento) && $v->tipo_pagamento == '06') {
                        $total += $v->total;
                    }
                }
            }
        }
        return $total;
    }

    /**
     * Calcula o total de vendas em boleto
     *
     * @param array $vendas Array de objetos de venda (NFCe, NFe, OS)
     * @return float Total de vendas em boleto
     */
    private function totalizaVendasBoleto($vendas)
    {
        $total = 0;
        foreach ($vendas as $v) {
            if ($v->estado != 'cancelado') {
                // Se for OS com pagamento boleto
                if ($v->tipo == 'OS' && isset($v->forma_pagamento) && $v->forma_pagamento == '15') {
                    $total += $v->valor;
                }
                // Se for NFCe ou NFe, verificar faturas
                else if ($v->tipo != 'OS') {
                    // Carregar faturas se necessário
                    $faturas = $v->fatura ?? FaturaNfce::where('nfce_id', $v->id)->get();

                    // Somar valores de faturas com tipo_pagamento = '15' (boleto)
                    if ($faturas && count($faturas) > 0) {
                        foreach ($faturas as $f) {
                            if ($f->tipo_pagamento == '15') {
                                $total += $f->valor;
                            }
                        }
                    }
                    // Se não tiver faturas mas tiver tipo_pagamento = '15', considerar o total
                    else if (isset($v->tipo_pagamento) && $v->tipo_pagamento == '15') {
                        $total += $v->total;
                    }
                }
            }
        }
        return $total;
    }

    /**
     * Calcula o total de vendas em crédito loja
     *
     * @param array $vendas Array de objetos de venda (NFCe, NFe, OS)
     * @return float Total de vendas em crédito loja
     */
    private function totalizaVendasCreditoLoja($vendas)
    {
        $total = 0;
        foreach ($vendas as $v) {
            if ($v->estado != 'cancelado') {
                // Se for OS com pagamento crédito loja
                if ($v->tipo == 'OS' && isset($v->forma_pagamento) && $v->forma_pagamento == '05') {
                    $total += $v->valor;
                }
                // Se for NFCe ou NFe, verificar faturas
                else if ($v->tipo != 'OS') {
                    // Carregar faturas se necessário
                    $faturas = $v->fatura ?? FaturaNfce::where('nfce_id', $v->id)->get();

                    // Somar valores de faturas com tipo_pagamento = '05' (crédito loja)
                    if ($faturas && count($faturas) > 0) {
                        foreach ($faturas as $f) {
                            if ($f->tipo_pagamento == '05') {
                                $total += $f->valor;
                            }
                        }
                    }
                    // Se não tiver faturas mas tiver tipo_pagamento = '05', considerar o total
                    else if (isset($v->tipo_pagamento) && $v->tipo_pagamento == '05') {
                        $total += $v->total;
                    }
                }
            }
        }
        return $total;
    }

    /**
     * Calcula o valor em caixa com a fórmula: Suprimentos + Vendas recebidas + Contas recebidas - Sangrias
     *
     * @param int $caixa_id ID do caixa
     * @return float Valor calculado em caixa
     */
    private function calcularValorEmCaixa($caixa_id)
    {
        // Buscar dados necessários
        $nfce = Nfce::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $caixa_id)
            ->with('itens') // Carregar itens para calcular valor total dos produtos
            ->get();

        $nfe = Nfe::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $caixa_id)
            ->where('tpNF', 1)
            ->where('orcamento', 0)
            ->with('itens') // Carregar itens para calcular valor total dos produtos
            ->get();

        $ordens = OrdemServico::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $caixa_id)
            ->get();

        // Verificar NFCes
        foreach ($nfce as $v) {
            // Calcular o valor total dos produtos
            $valorProdutos = 0;
            foreach ($v->itens as $item) {
                $valorProdutos += $item->sub_total;
            }

            // Verificar se há desconto e se foi aplicado corretamente
            if (isset($v->desconto) && $v->desconto > 0) {
                // Verificar se o desconto já está considerado no total
                $descontoAplicado = $valorProdutos - $v->total;
                if (abs($descontoAplicado - $v->desconto) > 0.5) {
                    // Atualizar o total para considerar o desconto
                    $v->total = $valorProdutos - $v->desconto;
                    Log::info('calcularValorEmCaixa: Ajustando total da NFCe #'.$v->id.' para '.$v->total.' com desconto de R$'.$v->desconto);
                }
            }
            // Se não houver desconto, verificar se o total bate com a soma dos itens
            else if (abs($valorProdutos - $v->total) > 0.5) {
                $v->total = $valorProdutos;
                Log::info('calcularValorEmCaixa: Ajustando total da NFCe #'.$v->id.' para '.$v->total.' baseado na soma dos itens');
            }
        }

        // Verificar NFes
        foreach ($nfe as $v) {
            // Calcular o valor total dos produtos
            $valorProdutos = 0;
            foreach ($v->itens as $item) {
                $valorProdutos += $item->sub_total;
            }

            // Verificar se há discrepância entre o total e a soma dos itens
            if (abs($valorProdutos - $v->total) > 0.5 && !isset($v->desconto)) {
                $v->total = $valorProdutos;
                Log::info('calcularValorEmCaixa: Ajustando total da NFe #'.$v->id.' para '.$v->total.' baseado na soma dos itens');
            }
        }

        // Agrupa os dados e calcula totais - passamos null para compras
        $data = $this->agrupaDados($nfce, $nfe, $ordens, null);

        // Calcular o total das vendas
        $totalVendas = 0;
        foreach($data as $v) {
            // Não precisamos verificar se não é compra, pois compras não estão mais no array
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendas += $valor;
            }
        }

        // Calcular vendas a crediário, boleto e crédito loja
        $totalCrediario = $this->totalizaVendasCrediario($data);
        $totalBoleto = $this->totalizaVendasBoleto($data);
        $totalCreditoLoja = $this->totalizaVendasCreditoLoja($data);

        // Calcular vendas recebidas (vendas efetivas)
        $totalVendasEfetivas = $totalVendas - $totalCrediario - $totalBoleto - $totalCreditoLoja;

        // Buscar suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $caixa_id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $caixa_id)->sum('valor');

        // Buscar contas recebidas
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $caixa_id)
            ->where('status', 1)
            ->get();

        // Calcular total de contas recebidas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            // Se tiver valor_recebido definido e for maior que zero, usar esse valor
            // Caso contrário, usar o valor_integral
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        // Buscar devoluções feitas no período deste caixa
        $item = Caixa::find($caixa_id);
        $devolucoesDinheiro = 0;
        $devolucoesPix = 0;

        if ($item) {
            try {
                // Consulta base
                $query = Devolucao::where('empresa_id', request()->empresa_id)
                    ->where('is_troca', false)
                    ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

                // Executar a consulta
                $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();

                // Somar devoluções em dinheiro (tipo_pagamento = 01) e PIX (tipo_pagamento = 17)
                foreach($devolucoes as $devolucao) {
                    if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                        $devolucoesDinheiro += $devolucao->valor_total;
                    } elseif ($devolucao->tipo_pagamento == '17') { // PIX
                        $devolucoesPix += $devolucao->valor_total;
                    }
                }
            } catch (\Exception $e) {
                // Logar o erro
                Log::error('Erro ao buscar devoluções no calcularValorEmCaixa: ' . $e->getMessage());
                // Manter $devolucoesDinheiro e $devolucoesPix como 0 em caso de erro
            }
        }

        // Buscar trocas realizadas no caixa
        $trocas = $this->getTrocasCaixa($caixa_id);

        // Buscar contas pagas no caixa
        $valorContasPagas = ContaPagar::where('empresa_id', request()->empresa_id)
            ->where('caixa_id', $caixa_id)
            ->where('status', 1)
            ->sum('valor_integral');

        // Calcular valor em caixa: Abertura + Suprimentos + Vendas recebidas + Contas recebidas + Trocas (entradas) - Sangrias - Devoluções em dinheiro/PIX - Trocas (saídas) - Contas Pagas
        $valorTotal = $item->valor_abertura + $somaSuprimento + $totalVendasEfetivas + $valorRecebidoTotal + $trocas->totalEntradas - $somaSangria - $devolucoesDinheiro - $devolucoesPix - $trocas->totalSaidas - $valorContasPagas;

        return $valorTotal;
    }

    public function fecharConta($id){
        $item = Caixa::findOrFail($id);

        $somaTiposPagamento = [];
        $contas = [];
        $nfce = Nfce::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)
        ->with('fatura')  // Carrega faturas com eager loading
        ->get();
        $nfe = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)->where('tpNF', 1)
        ->with('fatura')  // Carrega faturas com eager loading
        ->get();

        $ordens = OrdemServico::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)
        ->get();

        // Não carregamos mais as compras
        // $compras = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)->where('tpNF', 0)
        // ->where('orcamento', 0)
        // ->get();

        // Passamos null para compras em agrupaDados
        $data = $this->agrupaDados($nfce, $nfe, $ordens, null);
        $somaTiposPagamento = $this->somaTiposPagamento($data);
        $contasEmpresa = ContaEmpresa::where('empresa_id', request()->empresa_id)
        ->where('local_id', $item->local_id)
        ->where('status', 1)->get();

        // Calcula o valor em caixa para sugestão de fechamento
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        return view('caixa.fechar_lista', compact('item', 'somaTiposPagamento', 'contasEmpresa', 'valorEmCaixa'));
    }

    /**
     * Método para lidar com a rota POST /fechar
     * Realiza o fechamento direto do caixa a partir do modal
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function fechar(Request $request)
    {
        try {
            // Inicia uma transação para garantir a integridade dos dados
            return DB::transaction(function() use ($request) {
                // Obtém o ID do caixa a partir do formulário
                $id = $request->caixa_id;

                // Busca o caixa
                $item = Caixa::findOrFail($id);

                // Verifica se o caixa já está fechado
                if ($item->status == 0) {
                    session()->flash('flash_warning', 'Este caixa já está fechado!');
                    return redirect()->route('caixa.index');
                }

                // Valores do formulário do modal
                $valorFechamento = __convert_value_bd($request->valor_fechamento);
                $valorDinheiro = __convert_value_bd($request->valor_dinheiro);
                $valorCheque = __convert_value_bd($request->valor_cheque);
                $valorOutros = __convert_value_bd($request->valor_outros);
                $observacao = $request->observacao ?? 'Caixa fechado via modal';

                // Fecha o caixa
                $item->status = 0; // 0 = fechado
                $item->data_fechamento = now();
                $item->valor_fechamento = $valorFechamento;
                $item->valor_dinheiro = $valorDinheiro;
                $item->valor_cheque = $valorCheque;
                $item->valor_outros = $valorOutros;
                $item->observacao = $observacao;

                // Salva as alterações
                $item->save();

                // Registra log da operação
                $descricaoLog = $item->usuario->name . " | CAIXA FECHADO PELO MODAL - fechamento: " . __data_pt($item->updated_at) . " - valor fechamento: " . __moeda($item->valor_fechamento);
                __createLog(request()->empresa_id, 'Caixa', 'editar', $descricaoLog);

                // Redireciona com mensagem de sucesso
                session()->flash('flash_success', 'Caixa fechado com sucesso!');
                return redirect()->route('caixa.list');
            });
        } catch (\Exception $e) {
            // Em caso de erro, registra no log e retorna mensagem
            Log::error('Erro ao fechar caixa via modal: ' . $e->getMessage());
            session()->flash('flash_error', 'Erro ao fechar caixa: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Processa o formulário de fechamento do caixa e finaliza o caixa.
     *
     * @param Request $request
     * @param int $id ID do caixa
     * @return \Illuminate\Http\Response
     */
    public function fecharTiposPagamento(Request $request, $id)
    {
        try {
            // Inicia uma transação para garantir a integridade dos dados
            return DB::transaction(function() use ($request, $id) {
                // Busca o caixa
                $item = Caixa::findOrFail($id);

                // Verifica se o caixa já está fechado
                if ($item->status == 0) {
                    session()->flash('flash_warning', 'Este caixa já está fechado!');
                    return redirect()->route('caixa.index');
                }

                // Busca o valor em caixa atual para usar como valor_fechamento
                $valorEmCaixa = $this->calcularValorEmCaixa($id);

                // Fecha o caixa
                $item->status = 0; // 0 = fechado
                $item->data_fechamento = now();
                $item->valor_fechamento = $valorEmCaixa;

                // Para garantir que estamos mantendo a compatibilidade com o sistema existente
                // Verifica se existem valores nos tipos de pagamento tradicionais
                $somaDinheiro = 0;
                $somaCheque = 0;
                $somaOutros = 0;

                // Processa os tipos de pagamento enviados
                if ($request->tipo_pagamento) {
                    for ($i = 0; $i < count($request->tipo_pagamento); $i++) {
                        $tipoPagamento = $request->tipo_pagamento[$i];
                        $contaEmpresaId = $request->conta_empresa_id[$i];
                        $valor = __convert_value_bd($request->valor[$i]);
                        $descricao = $request->descricao[$i] ?? 'Fechamento de caixa';

                        // Classifica os valores para os campos tradicionais
                        if ($tipoPagamento == '01') { // Dinheiro
                            $somaDinheiro += $valor;
                        } else if ($tipoPagamento == '02') { // Cheque
                            $somaCheque += $valor;
                        } else {
                            $somaOutros += $valor;
                        }

                        // Cria um item de conta_empresa para registrar a movimentação
                        ItemContaEmpresa::create([
                            'conta_empresa_id' => $contaEmpresaId,
                            'empresa_id' => request()->empresa_id,
                            'valor' => $valor,
                            'descricao' => $descricao,
                            'tipo_pagamento' => $tipoPagamento,
                            'caixa_id' => $id,
                            'data_movimento' => now(),
                            'tipo' => 'receita'
                        ]);
                    }
                }

                // Atualiza os valores classificados
                $item->valor_dinheiro = $somaDinheiro;
                $item->valor_cheque = $somaCheque;
                $item->valor_outros = $somaOutros;

                // Salva as alterações
                $item->save();

                // Registra log da operação
                $descricaoLog = $item->usuario->name . " | CAIXA FECHADO - fechamento: " . __data_pt($item->updated_at) . " - valor fechamento: " . __moeda($item->valor_fechamento);
                __createLog(request()->empresa_id, 'Caixa', 'editar', $descricaoLog);

                // Redireciona com mensagem de sucesso
                session()->flash('flash_success', 'Caixa fechado com sucesso!');
                return redirect()->route('caixa.list');
            });
        } catch (\Exception $e) {
            // Em caso de erro, registra no log e retorna mensagem
            Log::error('Erro ao fechar caixa: ' . $e->getMessage());
            session()->flash('flash_error', 'Erro ao fechar caixa: ' . $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    /**
     * Busca as trocas realizadas no período do caixa
     *
     * @param int $caixa_id ID do caixa
     * @return \Illuminate\Database\Eloquent\Collection
     */
    private function getTrocasCaixa($caixa_id)
    {
        $resultado = [
            'trocas' => collect([]),
            'totalEntradas' => 0,
            'totalSaidas' => 0
        ];

        $item = Caixa::find($caixa_id);

        if ($item) {
            try {
                // Consulta base
                $query = Devolucao::where('empresa_id', request()->empresa_id)
                    ->where('is_troca', true) // Filtrar apenas trocas
                    ->where('caixa_id', $caixa_id); // Filtrar pelo ID do caixa

                // Executar a consulta com relacionamentos necessários
                $resultado['trocas'] = $query->with(['itens', 'usuario', 'cliente'])->get();

                // Calcular os totais
                foreach ($resultado['trocas'] as $troca) {
                    if ($troca->valor_diferenca < 0) {
                        // Valor negativo significa que o cliente pagou (entrada)
                        $resultado['totalEntradas'] += abs($troca->valor_diferenca);
                    } else if ($troca->valor_diferenca > 0) {
                        // Valor positivo significa que o cliente recebeu (saída)
                        $resultado['totalSaidas'] += $troca->valor_diferenca;
                    }
                }

            } catch (\Exception $e) {
                Log::error('Erro ao buscar trocas: ' . $e->getMessage());
            }
        }

        return (object) $resultado;
    }

    public function abertosEmpresa()
    {
        $empresaId = request()->empresa_id;
        $data = Caixa::whereHas('usuario', function ($query) use ($empresaId) {
                $query->where('empresa_id', $empresaId);
            })
            ->where('status', 1)
            ->with('usuario')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('caixa.list', compact('data'));
    }

    /**
     * Lista todos os caixas para o usuário atual ou para o administrador
     *
     * @return \Illuminate\View\View
     */
    public function list()
    {
        $empresaId = request()->empresa_id;
        $dataInicial = request()->get('data_inicial');
        $dataFinal = request()->get('data_final');

        // Query base
        $query = Caixa::query();

        // Se for administrador, mostra todos os caixas da empresa
        if (__isAdmin()) {
            $query->whereHas('usuario', function ($q) use ($empresaId) {
                $q->where('empresa_id', $empresaId);
            });
        }
        // Se não for administrador, mostra apenas os caixas do usuário atual
        else {
            $query->where('usuario_id', Auth::user()->id);
        }

        // Aplicar filtros de data se fornecidos
        if ($dataInicial) {
            $query->whereDate('created_at', '>=', $dataInicial);
        }

        if ($dataFinal) {
            $query->whereDate('created_at', '<=', $dataFinal);
        }

        $data = $query->with('usuario')
            ->orderBy('created_at', 'desc')
            ->get();

        return view('caixa.list', compact('data', 'dataInicial', 'dataFinal'));
    }

    /**
     * Gera um relatório em PDF para o caixa especificado.
     *
     * @param string $id ID do caixa
     * @return \Illuminate\Http\Response
     */
    public function imprimirRelatorio($id)
    {
        $item = Caixa::findOrFail($id);
        $valor_abertura = $item->valor_abertura;

        $nfce = Nfce::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $nfe = Nfe::where('empresa_id',  request()->empresa_id)->where('caixa_id', $item->id)
        ->where('tpNF', 1)
        ->where('orcamento', 0)
        ->with('fatura') // Eager loading das faturas
        ->get();

        $pagar = ContaPagar::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();
        $receber = ContaReceber::where('empresa_id', request()->empresa_id)->where('caixa_id', $item->id)->get();

        $data = $this->agrupaDados($nfce, $nfe, null, null);

        // Criar versão expandida dos dados, com pagamentos separados
        $dadosExpandidos = $this->expandeDadosPorPagamento($data);

        // Agrupar os dados expandidos por venda
        $vendasAgrupadas = $this->agrupaVendas($dadosExpandidos);

        // Processar contas a receber com múltiplas formas de pagamento
        $recebimentosExpandidos = $this->expandeContasReceberPorPagamento($receber);
        $recebimentosAgrupados = $this->agrupaContasReceber($recebimentosExpandidos);

        $somaTiposPagamento = $this->somaTiposPagamento($data);
        $contas = $this->agrupaContas($pagar, $receber);
        $somaTiposContas = $this->somaTiposContas($contas);

        $suprimentos = SuprimentoCaixa::where('caixa_id', $item->id)->get();
        $sangrias = SangriaCaixa::where('caixa_id', $item->id)->get();

        // Calcular total de serviços
        $somaServicos = ItemServicoNfce::join('nfces', 'nfces.id', '=', 'item_servico_nfces.nfce_id')
            ->where('nfces.empresa_id', request()->empresa_id)
            ->where('nfces.caixa_id', $item->id)
            ->sum('sub_total');

        // Buscar devoluções feitas neste caixa
        $devolucoes = [];
        try {
            // Consulta base
            $query = Devolucao::where('empresa_id', request()->empresa_id)
                ->where('is_troca', false)
                ->where('caixa_id', $item->id); // Filtrar apenas as devoluções deste caixa

            // Executar a consulta
            $devolucoes = $query->with(['itens', 'usuario', 'cliente'])->get();
        } catch (\Exception $e) {
            // Logar o erro
            Log::error('Erro ao buscar devoluções: ' . $e->getMessage());
            // Inicializar com array vazio em caso de erro
            $devolucoes = collect([]);
        }

        // Calcular valores de devoluções por tipo de pagamento
        $devolucoesPorTipo = [
            'dinheiro' => 0, // tipo_pagamento = 01
            'cartao' => 0,    // tipo_pagamento = 03, 04
            'credito' => 0,   // tipo_pagamento = 00
            'pix' => 0        // tipo_pagamento = 17
        ];

        $totalDevolucoes = 0;

        foreach($devolucoes as $devolucao) {
            $totalDevolucoes += $devolucao->valor_total;

            // Categorizar por tipo de pagamento
            if ($devolucao->tipo_pagamento == '01') { // Dinheiro
                $devolucoesPorTipo['dinheiro'] += $devolucao->valor_total;
            }
            elseif (in_array($devolucao->tipo_pagamento, ['03', '04'])) { // Cartões
                $devolucoesPorTipo['cartao'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '00') { // Crédito Loja
                $devolucoesPorTipo['credito'] += $devolucao->valor_total;
            }
            elseif ($devolucao->tipo_pagamento == '17') { // PIX
                $devolucoesPorTipo['pix'] += $devolucao->valor_total;
            }
        }

        // Ajustar o cálculo do valor final considerando as devoluções em dinheiro e PIX
        $valorEmCaixa = $this->calcularValorEmCaixa($item->id);

        // Subtrair as devoluções em dinheiro e PIX do valor em caixa
        $valorEmCaixa -= ($devolucoesPorTipo['dinheiro'] + $devolucoesPorTipo['pix']);

        // Buscar trocas realizadas no caixa
        $trocas = $this->getTrocasCaixa($item->id);

        $data_hoje = date('d/m/Y');
        $produtos = $this->totalizaProdutos($data);

        // Calcular o total das vendas
        $totalVendas = 0;
        foreach($data as $v) {
            if($v->estado != 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendas += $valor;
            }
        }

        // Calcular vendas a crediário, boleto e crédito loja
        $totalCrediario = $this->totalizaVendasCrediario($data);
        $totalBoleto = $this->totalizaVendasBoleto($data);
        $totalCreditoLoja = $this->totalizaVendasCreditoLoja($data);

        // Calcular vendas recebidas (vendas efetivas)
        $totalVendasEfetivas = $totalVendas - $totalCrediario - $totalBoleto - $totalCreditoLoja;

        // Buscar suprimentos e sangrias
        $somaSuprimento = SuprimentoCaixa::where('caixa_id', $item->id)->sum('valor');
        $somaSangria = SangriaCaixa::where('caixa_id', $item->id)->sum('valor');

        // Calcular total de contas recebidas
        $valorRecebidoTotal = 0;
        foreach($receber as $conta) {
            // Se tiver valor_recebido definido e for maior que zero, usar esse valor
            // Caso contrário, usar o valor_integral
            $valorRecebido = ($conta->valor_recebido && $conta->valor_recebido > 0)
                ? $conta->valor_recebido
                : $conta->valor_integral;

            $valorRecebidoTotal += $valorRecebido;
        }

        // Calcular vendas canceladas
        $totalVendasCanceladas = 0;
        foreach($data as $v) {
            if($v->estado == 'cancelado') {
                $valor = $v->tipo != 'OS' ? $v->total : $v->valor;
                $totalVendasCanceladas += $valor;
            }
        }

        // Adicionar o título para o relatório
        $title = "Relatório de Caixa #" . $id;

        // Buscar a empresa atual
        $empresa = Empresa::findOrFail(request()->empresa_id);

        // Adicionar informações de usuário, data de abertura e fechamento
        $usuario = User::findOrFail($item->usuario_id);
        $data_abertura = date('d/m/Y', strtotime($item->created_at));
        $abertura_hora = date('H:i:s', strtotime($item->created_at));

        $data_fechamento = $item->status == 1 ? 'Em aberto' : date('d/m/Y', strtotime($item->updated_at));
        $fechamento_hora = $item->status == 1 ? '--:--:--' : date('H:i:s', strtotime($item->updated_at));

        $dompdf = new Dompdf();
        $dompdf->loadHtml(view('caixa.imprimir', compact(
            'item',
            'data',
            'dadosExpandidos',
            'vendasAgrupadas',
            'somaTiposPagamento',
            'valor_abertura',
            'valorEmCaixa',
            'usuario',
            'data_abertura',
            'abertura_hora',
            'data_fechamento',
            'fechamento_hora',
            'suprimentos',
            'sangrias',
            'somaServicos',
            'contas',
            'somaTiposContas',
            'devolucoes',
            'devolucoesPorTipo',
            'totalDevolucoes',
            'data_hoje',
            'produtos',
            'recebimentosExpandidos',
            'recebimentosAgrupados',
            'totalVendas',
            'totalVendasCanceladas',
            'totalCrediario',
            'totalBoleto',
            'totalCreditoLoja',
            'totalVendasEfetivas',
            'somaSuprimento',
            'somaSangria',
            'valorRecebidoTotal',
            'pagar',
            'title',
            'trocas'
        ))->with('config', $empresa)->render());
        $dompdf->setPaper('A4', 'landscape');
        $dompdf->render();
        $dompdf->stream("Relatório de caixa.pdf", array("Attachment" => false));
    }
}

