<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\OrdemProducao;
use App\Models\ItemProducao;
use App\Models\ItemOrdemProducao;
use App\Models\Produto;
use App\Models\Cliente;
use App\Models\Empresa;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Barryvdh\DomPDF\Facade\Pdf as PDF;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\RelatorioProducaoExport;

class OrdemProducaoController extends Controller
{

    public function __construct()
    {
        $this->middleware('permission:ordem_producao_create', ['only' => ['create', 'store']]);
        $this->middleware('permission:ordem_producao_edit', ['only' => ['edit', 'update']]);
        $this->middleware('permission:ordem_producao_view', ['only' => ['show', 'index']]);
        $this->middleware('permission:ordem_producao_delete', ['only' => ['destroy']]);
    }

    // Pesquisa de clientes (rota web) para Select2
    public function searchClientes(Request $request)
    {
        $empresa_id = $request->empresa_id;
        $term = trim((string)$request->get('term', ''));

        $q = Cliente::where('empresa_id', $empresa_id);
        if ($term !== '') {
            $q->where(function($w) use ($term){
                $w->where('razao_social', 'like', "%{$term}%")
                  ->orWhere('nome_fantasia', 'like', "%{$term}%")
                  ->orWhere('cpf_cnpj', 'like', "%{$term}%");
            });
        }
        $rows = $q->orderByRaw('COALESCE(nome_fantasia, razao_social) ASC')
            ->limit(20)
            ->get(['id','razao_social','nome_fantasia','cpf_cnpj']);

        $results = $rows->map(function($c){
            $nome = $c->nome_fantasia ?: $c->razao_social;
            $doc = $c->cpf_cnpj ?: '';
            return [
                'id' => $c->id,
                'text' => 'ID:' . $c->id . ' | ' . $nome . ' | ' . $doc,
                'nome' => $nome,
                'cpf_cnpj' => $doc,
            ];
        });

        return response()->json(['results' => $results]);
    }

    public function imprimirPdf(Request $request, $id)
    {
        // Carrega a OP pelo ID e valida a empresa, quando informada
        $op = OrdemProducao::with(['itens.produto', 'cliente', 'creator', 'finisher'])
            ->findOrFail($id);

        if ($request->filled('empresa_id') && (int)$request->empresa_id !== (int)$op->empresa_id) {
            abort(404);
        }

        $empresa = Empresa::find($op->empresa_id);

        $pdf = PDF::loadView('ordem_producao.show_pdf', compact('op', 'empresa'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'isRemoteEnabled' => true,
                'defaultFont' => 'DejaVu Sans',
            ]);
        $filename = 'OP_' . $op->numero . '.pdf';
        return $pdf->download($filename);
    }

    public function relatorioPdf(Request $request)
    {
        $empresa_id = $request->empresa_id;
        if (empty($empresa_id)) { $empresa_id = session('empresa_id'); }
        if (empty($empresa_id)) { $empresa_id = Auth::user()->empresa_id ?? $empresa_id; }
        // Garante empresa em sessão para middlewares que dependem disso
        if (!empty($empresa_id)) { session(['empresa_id' => $empresa_id]); }
        $q = DB::table('item_ordem_producaos as i')
            ->join('ordem_producaos as o', 'o.id', '=', 'i.ordem_producao_id')
            ->join('produtos as p', 'p.id', '=', 'i.produto_id')
            ->leftJoin('clientes as c', 'c.id', '=', 'o.cliente_id')
            ->leftJoin('users as u', 'u.id', '=', 'o.created_by')
            ->where('o.empresa_id', $empresa_id);

        if ($request->filled('start_date')) { $q->whereDate('o.data_producao', '>=', $request->start_date); }
        if ($request->filled('end_date')) { $q->whereDate('o.data_producao', '<=', $request->end_date); }
        if ($request->filled('situacao')) { $q->where('o.situacao', $request->situacao); }
        if ($request->filled('cliente_id')) { $q->where('o.cliente_id', $request->cliente_id); }
        if ($request->filled('produto_id')) { $q->where('i.produto_id', $request->produto_id); }

        $q->select('o.codigo_sequencial','o.data_producao','p.nome as produto','p.unidade','i.quantidade', DB::raw('COALESCE(c.nome_fantasia, c.razao_social) as cliente'),'o.situacao','u.name as usuario')
          ->orderBy('o.data_producao','desc')->orderBy('o.id','desc');

        $rows = $q->get();
        $totalQuantidade = (float)$rows->sum('quantidade');

        $empresa = Empresa::find($empresa_id);

        $pdf = PDF::loadView('ordem_producao.relatorio_pdf', compact('rows','totalQuantidade','empresa'))
            ->setPaper('a4', 'portrait')
            ->setOptions([
                'isRemoteEnabled' => true,
                'defaultFont' => 'DejaVu Sans',
            ]);
        $filename = 'relatorio_producao_' . now()->format('Ymd_His') . '.pdf';
        return $pdf->download($filename);
    }

    public function relatorioExcel(Request $request)
    {
        $empresa_id = $request->empresa_id;
        if (empty($empresa_id)) { $empresa_id = session('empresa_id'); }
        if (empty($empresa_id)) { $empresa_id = Auth::user()->empresa_id ?? $empresa_id; }
        $filters = array_merge($request->all(), ['empresa_id' => $empresa_id]);
        $filename = 'relatorio_producao_' . now()->format('Ymd_His') . '.xlsx';
        return Excel::download(new RelatorioProducaoExport($filters, $empresa_id), $filename);
    }

    public function index(Request $request){
        $q = OrdemProducao::with(['cliente'])
            ->where('empresa_id', $request->empresa_id)
            ->orderBy('id', 'desc');

        // Filtros
        if ($request->filled('start_date')) {
            $q->whereDate('data_producao', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $q->whereDate('data_producao', '<=', $request->end_date);
        }
        if ($request->filled('situacao')) {
            $q->where('situacao', $request->situacao);
        }
        if ($request->filled('cliente_id')) {
            $q->where('cliente_id', $request->cliente_id);
        }

        $data = $q->paginate(env("PAGINACAO"));

        return view('ordem_producao.index', compact('data'));
    }

    public function create(Request $request){
        // Itens de produção pendentes opcionais (fluxo antigo)
        $itensPendentes = ItemProducao::where('produtos.empresa_id', $request->empresa_id)
            ->select('item_producaos.*')
            ->join('produtos', 'produtos.id', '=', 'item_producaos.produto_id')
            ->where('item_producaos.status', 0)
            ->get();

        $clientes = Cliente::where('empresa_id', $request->empresa_id)
            ->select('id', 'nome_fantasia', 'razao_social')
            ->orderByRaw('COALESCE(razao_social, nome_fantasia) ASC')
            ->get();
        // Preferir razao_social para manter consistência com o cadastro
        $clientesOptions = $clientes->mapWithKeys(function($c){
            return [$c->id => ($c->razao_social ?? $c->nome_fantasia)];
        });
        $produtos = Produto::where('empresa_id', $request->empresa_id)->orderBy('nome')->get();
        $produtosOptions = $produtos->pluck('nome', 'id');
        $produtosJs = $produtos->map(function($p){
            return [
                'id' => $p->id,
                'nome' => $p->nome,
                'unidade' => $p->unidade,
            ];
        })->values();

        // Próximo número da OP (OP-00001)
        $maxSeq = (int)(OrdemProducao::where('empresa_id', $request->empresa_id)->max('codigo_sequencial') ?? 0);
        $proximoSequencial = $maxSeq + 1;
        $proximoNumero = 'OP-' . str_pad((string)$proximoSequencial, 5, '0', STR_PAD_LEFT);

        return view('ordem_producao.create', [
            'data' => $itensPendentes,
            'clientes' => $clientes,
            'clientesOptions' => $clientesOptions,
            'produtos' => $produtos,
            'produtosJs' => $produtosJs,
            'produtosOptions' => $produtosOptions,
            'proximoNumero' => $proximoNumero,
        ]);
    }

    public function store(Request $request){
        $request->validate([
            'data_producao' => 'required|date',
            'cliente_id' => 'nullable|exists:clientes,id',
            'observacao' => 'nullable|string|max:100',
            'data_prevista_entrega' => 'nullable|date',
            'produtos' => 'nullable|array',
            'produtos.*.produto_id' => 'required_with:produtos|exists:produtos,id',
            'produtos.*.quantidade' => 'required_with:produtos|numeric|min:0.0001',
            'item_select' => 'nullable|array',
        ]);

        $empresa_id = $request->empresa_id;

        DB::beginTransaction();
        try {
            // Próximo número sequencial por empresa
            $nextSeq = (int)(OrdemProducao::where('empresa_id', $empresa_id)->max('codigo_sequencial')) + 1;

            $op = OrdemProducao::create([
                'empresa_id' => $empresa_id,
                'observacao' => $request->observacao,
                'estado' => 'producao', // compatibilidade
                'situacao' => 'em_producao',
                'data_prevista_entrega' => $request->data_prevista_entrega,
                'data_producao' => $request->data_producao,
                'cliente_id' => $request->cliente_id,
                'created_by' => Auth::id(),
                'codigo_sequencial' => $nextSeq,
            ]);

            $totalItens = 0;
            // Itens vindos do seletor de ItemProducao
            if ($request->filled('item_select')) {
                $itens = ItemProducao::whereIn('id', $request->item_select)->get();
                foreach ($itens as $i) {
                    ItemOrdemProducao::create([
                        'ordem_producao_id' => $op->id,
                        'item_producao_id' => $i->id,
                        'produto_id' => $i->produto_id,
                        'quantidade' => $i->quantidade,
                        'status' => 0,
                        'observacao' => null,
                    ]);
                    $totalItens++;
                }
            }

            // Itens manuais produtos[][produto_id, quantidade]
            if ($request->filled('produtos')) {
                foreach ($request->produtos as $row) {
                    if (!isset($row['produto_id']) || !isset($row['quantidade'])) continue;
                    if ((float)$row['quantidade'] <= 0) continue;
                    ItemOrdemProducao::create([
                        'ordem_producao_id' => $op->id,
                        'item_producao_id' => null,
                        'produto_id' => $row['produto_id'],
                        'quantidade' => (float)$row['quantidade'],
                        'status' => 0,
                        'observacao' => $row['observacao'] ?? null,
                    ]);
                    $totalItens++;
                }
            }

            if ($totalItens == 0) {
                DB::rollBack();
                return back()->withErrors('Informe pelo menos um item de produção.')->withInput();
            }

            DB::commit();
            return redirect()->route('ordem-producao.index')->with('success', 'Ordem de produção criada com sucesso.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors('Falha ao criar ordem: ' . $e->getMessage())->withInput();
        }
    }

    public function edit(Request $request, $id){
        $op = OrdemProducao::with(['itens.produto', 'cliente'])->where('empresa_id', $request->empresa_id)->findOrFail($id);
        if ($op->situacao === 'finalizado') {
            return redirect()->route('ordem-producao.index')->withErrors('Ordem finalizada não pode ser editada.');
        }
        $clientes = Cliente::where('empresa_id', $request->empresa_id)
            ->select('id', 'nome_fantasia', 'razao_social')
            ->orderByRaw('COALESCE(razao_social, nome_fantasia) ASC')
            ->get();
        // Preferir razao_social para manter consistência com o cadastro
        $clientesOptions = $clientes->mapWithKeys(function($c){
            return [$c->id => ($c->razao_social ?? $c->nome_fantasia)];
        });
        $produtos = Produto::where('empresa_id', $request->empresa_id)->orderBy('nome')->get();
        $produtosOptions = $produtos->pluck('nome', 'id');
        $produtosJs = $produtos->map(function($p){
            return [
                'id' => $p->id,
                'nome' => $p->nome,
                'unidade' => $p->unidade,
            ];
        })->values();
        return view('ordem_producao.edit', compact('op', 'clientes', 'clientesOptions', 'produtos', 'produtosJs', 'produtosOptions'));
    }

    public function update(Request $request, $id){
        $op = OrdemProducao::where('empresa_id', $request->empresa_id)->findOrFail($id);
        if ($op->situacao === 'finalizado') {
            return redirect()->route('ordem-producao.index')->withErrors('Ordem finalizada não pode ser editada.');
        }

        $request->validate([
            'data_producao' => 'required|date',
            'cliente_id' => 'nullable|exists:clientes,id',
            'observacao' => 'nullable|string|max:100',
            'data_prevista_entrega' => 'nullable|date',
            'produtos' => 'nullable|array',
            'produtos.*.produto_id' => 'required_with:produtos|exists:produtos,id',
            'produtos.*.quantidade' => 'required_with:produtos|numeric|min:0.0001',
        ]);

        DB::beginTransaction();
        try {
            // Atualiza cabeçalho
            $op->update([
                'data_producao' => $request->data_producao,
                'cliente_id' => $request->cliente_id,
                'observacao' => $request->observacao,
                'data_prevista_entrega' => $request->data_prevista_entrega,
            ]);

            // Substitui itens: remove atuais e recria conforme o formulário
            $op->itens()->delete();

            $totalItens = 0;
            if ($request->filled('produtos')) {
                foreach ($request->produtos as $row) {
                    if (!isset($row['produto_id']) || !isset($row['quantidade'])) continue;
                    $qtd = (float)$row['quantidade'];
                    if ($qtd <= 0) continue;
                    ItemOrdemProducao::create([
                        'ordem_producao_id' => $op->id,
                        'item_producao_id' => null,
                        'produto_id' => $row['produto_id'],
                        'quantidade' => $qtd,
                        'status' => 0,
                        'observacao' => $row['observacao'] ?? null,
                    ]);
                    $totalItens++;
                }
            }

            if ($totalItens == 0) {
                DB::rollBack();
                return back()->withErrors('Informe pelo menos um item de produção.')->withInput();
            }

            DB::commit();
            return redirect()->route('ordem-producao.index')->with('success', 'Ordem de produção atualizada.');
        } catch (\Throwable $e) {
            DB::rollBack();
            return back()->withErrors('Falha ao atualizar ordem: ' . $e->getMessage())->withInput();
        }
    }

    public function destroy(Request $request, $id){
        $op = OrdemProducao::where('empresa_id', $request->empresa_id)->findOrFail($id);
        if ($op->situacao === 'finalizado') {
            return redirect()->route('ordem-producao.index')->withErrors('Ordem finalizada não pode ser excluída.');
        }
        $op->itens()->delete();
        $op->delete();
        return redirect()->route('ordem-producao.index')->with('success', 'Ordem de produção excluída.');
    }

    public function finalizar(Request $request, $id){
        $op = OrdemProducao::where('empresa_id', $request->empresa_id)->findOrFail($id);
        if ($op->situacao !== 'em_producao') {
            return redirect()->route('ordem-producao.index')->withErrors('Apenas ordens em produção podem ser finalizadas.');
        }
        $op->update([
            'situacao' => 'finalizado',
            'finished_at' => now(),
            'finished_by' => Auth::id(),
        ]);
        return redirect()->route('ordem-producao.index')->with('success', 'Ordem finalizada com sucesso.');
    }

    public function show(Request $request, $id){
        $op = OrdemProducao::with(['itens.produto', 'cliente', 'creator', 'finisher'])
            ->where('empresa_id', $request->empresa_id)
            ->findOrFail($id);
        $empresa = Empresa::find($request->empresa_id);
        return view('ordem_producao.show', compact('op', 'empresa'));
    }

    public function relatorio(Request $request)
    {
        $empresa_id = $request->empresa_id;
        if (empty($empresa_id)) {
            $empresa_id = Auth::user()->empresa_id ?? $empresa_id;
        }

        $q = DB::table('item_ordem_producaos as i')
            ->join('ordem_producaos as o', 'o.id', '=', 'i.ordem_producao_id')
            ->join('produtos as p', 'p.id', '=', 'i.produto_id')
            ->leftJoin('clientes as c', 'c.id', '=', 'o.cliente_id')
            ->leftJoin('users as u', 'u.id', '=', 'o.created_by')
            ->where('o.empresa_id', $empresa_id);

        // Filtros
        if ($request->filled('start_date')) {
            $q->whereDate('o.data_producao', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $q->whereDate('o.data_producao', '<=', $request->end_date);
        }
        if ($request->filled('situacao')) {
            $q->where('o.situacao', $request->situacao);
        }
        if ($request->filled('cliente_id')) {
            $q->where('o.cliente_id', $request->cliente_id);
        }
        if ($request->filled('produto_id')) {
            $q->where('i.produto_id', $request->produto_id);
        }

        $q->select(
            'o.id as ordem_id', 'o.data_producao', 'o.situacao', 'o.codigo_sequencial',
            'p.nome as produto', 'p.unidade', 'i.quantidade',
            DB::raw('COALESCE(c.nome_fantasia, c.razao_social) as cliente'), 'u.name as usuario'
        )->orderBy('o.data_producao', 'desc')->orderBy('o.id', 'desc');

        $data = $q->paginate(env('PAGINACAO'));

        // Totalização com os mesmos filtros
        $qTotal = DB::table('item_ordem_producaos as i')
            ->join('ordem_producaos as o', 'o.id', '=', 'i.ordem_producao_id')
            ->where('o.empresa_id', $empresa_id);
        if ($request->filled('start_date')) {
            $qTotal->whereDate('o.data_producao', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $qTotal->whereDate('o.data_producao', '<=', $request->end_date);
        }
        if ($request->filled('situacao')) {
            $qTotal->where('o.situacao', $request->situacao);
        }
        if ($request->filled('cliente_id')) {
            $qTotal->where('o.cliente_id', $request->cliente_id);
        }
        if ($request->filled('produto_id')) {
            $qTotal->where('i.produto_id', $request->produto_id);
        }
        $totalQuantidade = (float)$qTotal->sum('i.quantidade');

        // Listas para selects
        $clientes = Cliente::where('empresa_id', $empresa_id)
            ->selectRaw('id, COALESCE(nome_fantasia, razao_social) as nome')
            ->orderBy('nome')
            ->pluck('nome', 'id');
        $produtos = Produto::where('empresa_id', $empresa_id)->orderBy('nome')->pluck('nome', 'id');

        return view('ordem_producao.relatorio', compact('data', 'totalQuantidade', 'clientes', 'produtos'));
    }

    public function relatorioPrint(Request $request)
    {
        $empresa_id = $request->empresa_id;
        if (empty($empresa_id)) {
            $empresa_id = Auth::user()->empresa_id ?? $empresa_id;
        }

        $q = DB::table('item_ordem_producaos as i')
            ->join('ordem_producaos as o', 'o.id', '=', 'i.ordem_producao_id')
            ->join('produtos as p', 'p.id', '=', 'i.produto_id')
            ->leftJoin('clientes as c', 'c.id', '=', 'o.cliente_id')
            ->leftJoin('users as u', 'u.id', '=', 'o.created_by')
            ->where('o.empresa_id', $empresa_id);

        if ($request->filled('start_date')) {
            $q->whereDate('o.data_producao', '>=', $request->start_date);
        }
        if ($request->filled('end_date')) {
            $q->whereDate('o.data_producao', '<=', $request->end_date);
        }
        if ($request->filled('situacao')) {
            $q->where('o.situacao', $request->situacao);
        }
        if ($request->filled('cliente_id')) {
            $q->where('o.cliente_id', $request->cliente_id);
        }
        if ($request->filled('produto_id')) {
            $q->where('i.produto_id', $request->produto_id);
        }

        $q->select(
            'o.id as ordem_id', 'o.data_producao', 'o.situacao', 'o.codigo_sequencial',
            'p.nome as produto', 'p.unidade', 'i.quantidade',
            DB::raw('COALESCE(c.nome_fantasia, c.razao_social) as cliente'), 'u.name as usuario'
        )->orderBy('o.data_producao', 'desc')->orderBy('o.id', 'desc');

        $rows = $q->get();
        $totalQuantidade = (float)$rows->sum('quantidade');
        $empresa = Empresa::find($empresa_id);

        return view('ordem_producao.relatorio_print', compact('rows', 'totalQuantidade', 'empresa'));
    }

    public function relatorioExport(Request $request)
    {
        $empresa_id = $request->empresa_id;
        if (empty($empresa_id)) {
            $empresa_id = Auth::user()->empresa_id ?? $empresa_id;
        }

        $q = DB::table('item_ordem_producaos as i')
            ->join('ordem_producaos as o', 'o.id', '=', 'i.ordem_producao_id')
            ->join('produtos as p', 'p.id', '=', 'i.produto_id')
            ->leftJoin('clientes as c', 'c.id', '=', 'o.cliente_id')
            ->leftJoin('users as u', 'u.id', '=', 'o.created_by')
            ->where('o.empresa_id', $empresa_id);

        if ($request->filled('start_date')) { $q->whereDate('o.data_producao', '>=', $request->start_date); }
        if ($request->filled('end_date')) { $q->whereDate('o.data_producao', '<=', $request->end_date); }
        if ($request->filled('situacao')) { $q->where('o.situacao', $request->situacao); }
        if ($request->filled('cliente_id')) { $q->where('o.cliente_id', $request->cliente_id); }
        if ($request->filled('produto_id')) { $q->where('i.produto_id', $request->produto_id); }

        $q->select(
            'o.data_producao', 'p.nome as produto', 'p.unidade', 'i.quantidade',
            DB::raw('COALESCE(c.nome_fantasia, c.razao_social) as cliente'), 'o.situacao', 'u.name as usuario', 'o.codigo_sequencial'
        )->orderBy('o.data_producao', 'desc')->orderBy('o.id', 'desc');

        $rows = $q->get();

        $filename = 'relatorio_producao_' . now()->format('Ymd_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        return response()->streamDownload(function () use ($rows) {
            $out = fopen('php://output', 'w');
            fputcsv($out, ['Data Produção', 'Produto', 'UN', 'Quantidade', 'Cliente', 'Situação', 'Usuário', 'Número'], ';');
            foreach ($rows as $r) {
                fputcsv($out, [
                    $r->data_producao, $r->produto, $r->unidade, $r->quantidade, $r->cliente, $r->situacao, $r->usuario, $r->codigo_sequencial
                ], ';');
            }
            fclose($out);
        }, $filename, $headers);
    }

    // Pesquisa AJAX de produtos para Select2
    public function searchProdutos(Request $request)
    {
        $empresa_id = $request->empresa_id;
        $term = trim((string)$request->get('term', ''));

        $q = Produto::where('empresa_id', $empresa_id);
        if ($term !== '') {
            $q->where('nome', 'like', "%{$term}%");
        }
        $prods = $q->orderBy('nome')->limit(20)->get(['id', 'nome', 'unidade']);

        return response()->json([
            'results' => $prods->map(function($p){
                return [
                    'id' => $p->id,
                    'text' => $p->nome,
                    'unidade' => $p->unidade,
                ];
            }),
        ]);
    }
}
