<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Imports\ProdutoImport;
use Maatwebsite\Excel\Facades\Excel;
use App\Models\Ncm;
use Illuminate\Support\Facades\DB;

class NcmController extends Controller
{
    public function index(Request $request){
        $this->validaNcm();
        $data = Ncm::
        when(!empty($request->descricao), function ($q) use ($request) {
            return $q->where('descricao', 'LIKE', "%$request->descricao%");
        })->paginate('50');
        return view('ncm.index', compact('data'));
    }

    public function create()
    {
        return view('ncm.create');
    }

    public function edit($id)
    {
        $item = Ncm::findOrFail($id);

        return view('ncm.edit', compact('item'));
    }

    public function store(Request $request)
    {
        try {

            $request->merge([
                'descricao' => "$request->codigo - $request->descricao"
            ]);

            Ncm::create($request->all());
            session()->flash("flash_success", "NCM criado com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
        }
        return redirect()->route('ncm.index');
    }

    public function update(Request $request, $id)
    {
        $item = Ncm::findOrFail($id);
        try {
            
            $item->fill($request->all())->save();
            session()->flash("flash_success", "NCM alterado com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());

        }
        return redirect()->route('ncm.index');
    }

    public function destroy($id)
    {
        $item = Ncm::findOrFail($id);
        try {
            $item->delete();
            session()->flash("flash_success", "NCM removido com sucesso!");
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Algo deu errado: '. $e->getMessage());
        }
        return redirect()->route('ncm.index');
    }
    
    /**
     * Exibe a página para importação de NCMs
     */
    public function importView()
    {
        return view('ncm.import');
    }
    
    /**
     * Processa a importação do arquivo Excel
     */
    public function importStore(Request $request)
    {
        ini_set('max_execution_time', 600); // Aumentar para 10 minutos
        ini_set('memory_limit', '512M'); // Aumentar o limite de memória
        
        try {
            $request->validate([
                'arquivo_excel' => 'required|mimes:xlsx|max:10240', // Aumentando para 10MB
            ], [
                'arquivo_excel.required' => 'O arquivo é obrigatório.',
                'arquivo_excel.mimes' => 'O arquivo deve ser do tipo XLSX.',
                'arquivo_excel.max' => 'O tamanho máximo do arquivo é 10MB.',
            ]);
            
            // Converter o Excel para um array antes de começar a transação
            $rows = Excel::toArray(new \stdClass(), $request->file('arquivo_excel'))[0];
            
            // Limpar todos os NCMs existentes (fora da transação)
            Ncm::truncate();
            
            // Processar em lotes para evitar transações muito longas
            $totalImportados = 0;
            $tamanhoLote = 500; // Reduzir para 500 registros por vez para maior confiabilidade
            $lotes = array_chunk($rows, $tamanhoLote);
            
            // Registrar o total de linhas no arquivo
            $totalLinhas = count($rows);
            $linhasProcessadas = 0;
            
            foreach ($lotes as $indiceLote => $lote) {
                DB::beginTransaction();
                
                try {
                    foreach ($lote as $row) {
                        $linhasProcessadas++;
                        
                        // Ignorar a primeira linha por ser cabeçalho
                        if ($linhasProcessadas == 1 && isset($row[0]) && ($row[0] == 'Código' || $row[0] == 'Código')) {
                            continue;
                        }

                        // Se os dois primeiros campos estiverem vazios, pule
                        if ((!isset($row[0]) || trim((string)$row[0]) === '') && 
                            (!isset($row[1]) || trim((string)$row[1]) === '')) {
                            continue;
                        }

                        $codigo = isset($row[0]) ? trim((string)$row[0]) : '';
                        $descricao = isset($row[1]) ? trim((string)$row[1]) : '';

                        // Se o código estiver vazio mas a descrição não, pode ser um item complementar
                        // Ignorar apenas se ambos estiverem vazios
                        if ($codigo === '' && $descricao === '') {
                            continue;
                        }

                        // Para entradas que começam com números, limpar os caracteres especiais
                        if (is_numeric(substr($codigo, 0, 1))) {
                            $descricao = str_replace("--", "", $descricao);
                            $descricao = str_replace("-", "", $descricao);
                        }

                        Ncm::create([
                            'codigo' => $codigo,
                            'descricao' => "$codigo - $descricao"
                        ]);
                        
                        $totalImportados++;
                    }
                    
                    DB::commit();
                } catch (\Exception $e) {
                    DB::rollBack();
                    throw new \Exception("Erro no lote " . ($indiceLote + 1) . ": " . $e->getMessage());
                }
            }
            
            // Verificar se todas as linhas foram processadas
            if ($totalImportados < ($totalLinhas - 1)) { // -1 para o cabeçalho
                session()->flash("flash_warning", "$totalImportados NCMs importados com sucesso! Nota: O arquivo tinha aproximadamente " . ($totalLinhas - 1) . " linhas, mas algumas podem ter sido ignoradas por estarem vazias ou em formato inválido.");
            } else {
                session()->flash("flash_success", "$totalImportados NCMs importados com sucesso!");
            }
            
        } catch (\Exception $e) {
            session()->flash("flash_error", 'Falha na importação: '. $e->getMessage());
        }
        
        return redirect()->route('ncm.index');
    }

    private function validaNcm(){
        if(!file_exists(public_path('tabela_ncm.xlsx'))){
            abort("403", "Arquivo tabela_ncm.xlsx não encontrado!");
        }
        $data = Ncm::count();
        if($data == 0){

            $file = file_get_contents(public_path('tabela_ncm.xlsx'));

            $rows = Excel::toArray(new ProdutoImport, public_path('tabela_ncm.xlsx'));

            foreach($rows[0] as $key => $line){

                if(isset($line[0])){
                    $codigo = trim($line[0]);
                    $descricao = trim($line[1]);

                    if($key > 4){
                        $descricao = str_replace("--", "", $descricao);
                        $descricao = str_replace("-", "", $descricao);

                        Ncm::create([
                            'codigo' => $codigo,
                            'descricao' => "$codigo - $descricao"
                        ]);
                    }
                }
            }
        }
    }
}
