<?php

namespace App\Http\Controllers\API\Token;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\MesaComanda;

class MesaComandaController extends Controller
{
    protected $prefix = 'comandas';

    public function index(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".read")) {
            return response()->json("Permissão negada!", 403);
        }

        $data = MesaComanda::where('empresa_id', $request->empresa_id)
            ->select('id', 'empresa_id', 'numero', 'status', 'created_at', 'updated_at')
            ->orderBy('numero', 'asc')
            ->get();

        __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'read', $this->prefix);
        return response()->json($data, 200);
    }

    public function find(Request $request, $id)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".read")) {
            return response()->json("Permissão negada!", 403);
        }

        $item = MesaComanda::where('empresa_id', $request->empresa_id)
            ->select('id', 'empresa_id', 'numero', 'status', 'created_at', 'updated_at')
            ->findOrFail($id);

        __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'read', $this->prefix);
        return response()->json($item, 200);
    }

    public function store(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".create")) {
            return response()->json("Permissão negada!", 403);
        }
        try {
            $this->validate($request, [
                'empresa_id' => 'required|integer',
                'numero' => 'required|integer',
                'status' => 'required|in:livre,ocupada'
            ]);

            // Garantir unicidade por empresa
            $existe = MesaComanda::where('empresa_id', $request->empresa_id)
                ->where('numero', $request->numero)
                ->first();
            if ($existe) {
                return response()->json('Número de comanda já existente para esta empresa.', 422);
            }

            $item = MesaComanda::create([
                'empresa_id' => $request->empresa_id,
                'numero' => $request->numero,
                'status' => $request->status,
            ]);

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'create', $this->prefix);
            return response()->json($item, 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'create', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }

    public function abrir(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".update")) {
            return response()->json("Permissão negada!", 403);
        }
        $this->validate($request, [
            'empresa_id' => 'required|integer',
            'id' => 'nullable|integer',
            'numero' => 'nullable|integer',
            'status' => 'required|in:ocupada',
        ]);

        try {
            // Localiza uma comanda LIVRE por id ou numero
            $query = MesaComanda::where('empresa_id', $request->empresa_id)
                ->where('status', 'livre');

            if ($request->filled('id')) {
                $query->where('id', $request->id);
            } elseif ($request->filled('numero')) {
                $query->where('numero', $request->numero);
            } else {
                return response()->json('Informe id ou numero da comanda.', 422);
            }

            $mesa = $query->first();
            if (!$mesa) {
                return response()->json('Comanda não disponível (não encontrada ou não está livre).', 422);
            }

            // Abre a comanda (marca como ocupada)
            $mesa->status = 'ocupada';
            $mesa->save();

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'update', $this->prefix);
            // Retorna apenas os dados da comanda (itens são manipulados por endpoints específicos)
            return response()->json($mesa, 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'update', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }

    public function update(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".update")) {
            return response()->json("Permissão negada!", 403);
        }
        try {
            $this->validate($request, [
                'id' => 'required|integer',
                'status' => 'nullable|in:livre,ocupada',
                'numero' => 'nullable|integer',
                'empresa_id' => 'required|integer',
            ]);

            $item = MesaComanda::where('empresa_id', $request->empresa_id)
                ->findOrFail($request->id);

            if ($request->has('numero')) {
                $existe = MesaComanda::where('empresa_id', $request->empresa_id)
                    ->where('numero', $request->numero)
                    ->where('id', '!=', $request->id)
                    ->first();
                if ($existe) {
                    return response()->json('Número de comanda já existente para esta empresa.', 422);
                }
            }

            $item->fill($request->only(['numero', 'status']))->save();

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'update', $this->prefix);
            return response()->json($item, 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'update', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }

    public function delete(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".delete")) {
            return response()->json("Permissão negada!", 403);
        }
        try {
            $this->validate($request, [
                'empresa_id' => 'required|integer',
                'numero' => 'required|integer',
            ]);

            $item = MesaComanda::where('empresa_id', $request->empresa_id)
                ->where('numero', $request->numero)
                ->first();
            if ($item == null) {
                return response()->json("Comanda não encontrada!", 403);
            }

            // Cancelar: limpa itens e marca como livre, sem excluir a comanda
            $item->itens()->delete();
            $item->status = 'livre';
            $item->save();

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'delete', $this->prefix);
            return response()->json($item->fresh(), 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'delete', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }

    public function itens(Request $request, $id)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".read")) {
            return response()->json("Permissão negada!", 403);
        }

        $mesa = MesaComanda::where('empresa_id', $request->empresa_id)
            ->findOrFail($id);

        $itens = $mesa->itens()->with('produto')->get();

        __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'read', $this->prefix);
        return response()->json($itens, 200);
    }

    public function storeItem(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".create")) {
            return response()->json("Permissão negada!", 403);
        }
        try {
            $this->validate($request, [
                'mesa_comanda_id' => 'required|integer',
                'produto_id' => 'required|integer',
                'quantidade' => 'required|numeric|min:0.01',
                'valor_unitario' => 'nullable|numeric|min:0',
            ]);

            $mesa = MesaComanda::where('empresa_id', $request->empresa_id)
                ->findOrFail($request->mesa_comanda_id);

            // Busca valor do produto caso não enviado
            $valorUnitario = $request->valor_unitario;
            if ($valorUnitario === null) {
                $produto = \App\Models\Produto::findOrFail($request->produto_id);
                $valorUnitario = $produto->valor_unitario;
            }
            $subtotal = round(((float)$request->quantidade) * ((float)$valorUnitario), 2);

            $item = new \App\Models\ItemComanda();
            $item->empresa_id = $request->empresa_id;
            $item->mesa_comanda_id = $mesa->id;
            $item->produto_id = $request->produto_id;
            $item->quantidade = $request->quantidade;
            $item->valor_unitario = $valorUnitario;
            $item->subtotal = $subtotal;
            $item->save();

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'create', $this->prefix);
            return response()->json($item->load('produto'), 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'create', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }

    public function deleteItem(Request $request)
    {
        if (!__validaPermissaoToken($request->token, $this->prefix . ".delete")) {
            return response()->json("Permissão negada!", 403);
        }
        try {
            $this->validate($request, [
                'id' => 'required|integer',
                'empresa_id' => 'required|integer',
            ]);

            $item = \App\Models\ItemComanda::where('empresa_id', $request->empresa_id)
                ->find($request->id);
            if (!$item) {
                return response()->json('Item não encontrado!', 404);
            }
            $item->delete();

            __createApiLog($request->empresa_id, $request->token, 'sucesso', '', 'delete', $this->prefix);
            return response()->json(['success' => true], 200);
        } catch (\Exception $e) {
            __createApiLog($request->empresa_id, $request->token, 'erro', $e->getMessage(), 'delete', $this->prefix);
            return response()->json("Algo deu errado: " . $e->getMessage(), 403);
        }
    }
}
