<?php
require_once 'config/database.php';
session_start();

if (!isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit();
}

// Definir timezone
date_default_timezone_set('America/Sao_Paulo');

// Inicializar conexão
$database = new Database();
$db = $database->getConnection();

// Processar filtros
$data_inicial = isset($_GET['data_inicial']) ? $_GET['data_inicial'] : date('Y-m-01');
$data_final = isset($_GET['data_final']) ? $_GET['data_final'] : date('Y-m-t');

// Buscar lançamentos com filtros
$params = [];
$where = [];

// Se não for admin, filtrar apenas os lançamentos do usuário
if (!isset($_SESSION['is_admin']) || $_SESSION['is_admin'] != 1) {
    $where[] = "l.usuario_id = :usuario_id";
    $params[':usuario_id'] = $_SESSION['user_id'];
}

if ($data_inicial) {
    $where[] = "l.data_lancamento >= :data_inicial";
    $params[':data_inicial'] = $data_inicial;
}

if ($data_final) {
    $where[] = "l.data_lancamento <= :data_final";
    $params[':data_final'] = $data_final;
}

$where_clause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

$query = "SELECT l.*, f.nome as forma_pagamento, 
                 DATE_FORMAT(l.data_lancamento, '%d/%m/%Y') as data_formatada
          FROM lancamentos l 
          JOIN formas_pagamento f ON l.forma_pagamento_id = f.id 
          $where_clause
          ORDER BY l.data_lancamento ASC, l.id ASC";

$stmt = $db->prepare($query);
foreach ($params as $key => $value) {
    $stmt->bindValue($key, $value);
}
$stmt->execute();
$lancamentos = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Calcular totais por forma de pagamento
$totais = [
    'entrada' => [
        'dinheiro' => 0,
        'outros' => 0,
        'total' => 0
    ],
    'saida' => [
        'dinheiro' => 0,
        'outros' => 0,
        'total' => 0
    ]
];

foreach ($lancamentos as $lancamento) {
    $tipo = $lancamento['entrada'] > 0 ? 'entrada' : 'saida';
    $valor = $tipo === 'entrada' ? $lancamento['entrada'] : $lancamento['saida'];
    
    if (strtolower($lancamento['forma_pagamento']) === 'dinheiro') {
        $totais[$tipo]['dinheiro'] += $valor;
    } else {
        $totais[$tipo]['outros'] += $valor;
    }
    $totais[$tipo]['total'] += $valor;
}

$saldo_geral = $totais['entrada']['total'] - $totais['saida']['total'];
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Relatório de Fluxo de Caixa</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        @media print {
            .no-print {
                display: none;
            }
            .card {
                border: none !important;
            }
            .card-header {
                background-color: white !important;
            }
            @page {
                margin: 1cm;
            }
        }
        .table th, .table td {
            padding: 0.5rem;
        }
    </style>
</head>
<body class="bg-light">
    <div class="container mt-4 mb-4">
        <div class="card">
            <div class="card-header text-center">
                <h4 class="mb-0">RELATÓRIO DE FLUXO DE CAIXA</h4>
                <small>
                    Período: <?php echo date('d/m/Y', strtotime($data_inicial)); ?> a <?php echo date('d/m/Y', strtotime($data_final)); ?><br>
                    Data de Emissão: <?php echo date('d/m/Y H:i'); ?><br>
                    Emitido por: <?php echo isset($_SESSION['user_name']) ? htmlspecialchars($_SESSION['user_name']) : ''; ?>
                </small>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th>Data</th>
                                <th>Descrição</th>
                                <th>Categoria</th>
                                <th>Forma de Pagamento</th>
                                <th class="text-end">Valor</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lancamentos as $lancamento): ?>
                            <tr>
                                <td><?php echo $lancamento['data_formatada']; ?></td>
                                <td><?php echo htmlspecialchars($lancamento['descricao']); ?></td>
                                <td><?php echo $lancamento['entrada'] > 0 ? 'Entrada' : 'Saída'; ?></td>
                                <td><?php echo htmlspecialchars($lancamento['forma_pagamento']); ?></td>
                                <td class="text-end <?php echo $lancamento['entrada'] > 0 ? 'text-success' : 'text-danger'; ?>">
                                    R$ <?php echo number_format($lancamento['entrada'] > 0 ? $lancamento['entrada'] : $lancamento['saida'], 2, ',', '.'); ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>

                <!-- Resumo dos Totais -->
                <div class="mt-4">
                    <div class="row">
                        <!-- Entradas -->
                        <div class="col-md-6">
                            <table class="table table-bordered">
                                <tbody>
                                    <tr>
                                        <td colspan="2" class="table-success"><strong>ENTRADAS</strong></td>
                                    </tr>
                                    <tr>
                                        <td>Banco/Cartão/Pix</td>
                                        <td class="text-end">R$ <?php echo number_format($totais['entrada']['outros'], 2, ',', '.'); ?></td>
                                    </tr>
                                    <tr>
                                        <td>Dinheiro</td>
                                        <td class="text-end">R$ <?php echo number_format($totais['entrada']['dinheiro'], 2, ',', '.'); ?></td>
                                    </tr>
                                    <tr class="table-success">
                                        <td><strong>Total de Entradas</strong></td>
                                        <td class="text-end"><strong>R$ <?php echo number_format($totais['entrada']['total'], 2, ',', '.'); ?></strong></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                        
                        <!-- Saídas -->
                        <div class="col-md-6">
                            <table class="table table-bordered">
                                <tbody>
                                    <tr>
                                        <td colspan="2" class="table-danger"><strong>SAÍDAS</strong></td>
                                    </tr>
                                    <tr>
                                        <td>Banco/Cartão/Pix</td>
                                        <td class="text-end">R$ <?php echo number_format($totais['saida']['outros'], 2, ',', '.'); ?></td>
                                    </tr>
                                    <tr>
                                        <td>Dinheiro</td>
                                        <td class="text-end">R$ <?php echo number_format($totais['saida']['dinheiro'], 2, ',', '.'); ?></td>
                                    </tr>
                                    <tr class="table-danger">
                                        <td><strong>Total de Saídas</strong></td>
                                        <td class="text-end"><strong>R$ <?php echo number_format($totais['saida']['total'], 2, ',', '.'); ?></strong></td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                    
                    <!-- Resultado -->
                    <table class="table table-bordered">
                        <tbody>
                            <tr class="<?php echo $saldo_geral >= 0 ? 'table-success' : 'table-danger'; ?>">
                                <td><strong>RESULTADO GERAL</strong></td>
                                <td class="text-end" style="width: 50%;"><strong>R$ <?php echo number_format($saldo_geral, 2, ',', '.'); ?></strong></td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        
        <!-- Botões de Ação -->
        <div class="mt-3 text-center no-print">
            <a href="lancamentos.php" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Voltar
            </a>
            <button onclick="window.print()" class="btn btn-dark">
                <i class="fas fa-print"></i> Imprimir
            </button>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
